/*******************************************************************************
 * Copyright (c) 2013 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.graphfile.adapters;

import java.util.HashMap;
import java.util.Map;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.common.request.FreshEscapedName;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.PossibleModel;
import org.simantics.graph.db.IImportAdvisor;
import org.simantics.graph.representation.Root;
import org.simantics.graphfile.ontology.GraphFileResource;
import org.simantics.layer0.Layer0;

public class SystemResourcePasteImportAdvisor implements IImportAdvisor {

	protected Resource root;
	protected Resource library;
	protected Resource model;
	protected String singleType = null;
	protected Map<String, String> nameMappings = new HashMap<String, String>();
	
	public SystemResourcePasteImportAdvisor(ReadGraph graph, Resource library) throws DatabaseException {
		this.library = library;
		this.model = graph.syncRequest(new PossibleModel(library));
	}

	public Resource getTarget() {
		return library;
	}

	public Resource analyzeType(ReadGraph graph, Root root) throws DatabaseException {
		return null;
	}
	
	@Override
	public Resource analyzeRoot(ReadGraph graph, Root root) throws DatabaseException {
		
		if("%model".equals(root.name)) return model;
		
		analyzeType(graph, root);

		String type = root.type;
		if(singleType != null) {
			if(!type.equals(singleType)) throw new DatabaseException("Paste of a set of different types of objects is not supported.");
		} else {
			singleType = type;
		}
		
		String newName = newName(graph, library, root.name);
		nameMappings.put(root.name, newName);
		
		return null;

	}
	
	public String newName(ReadGraph graph, Resource library, String name) throws DatabaseException {
		return graph.syncRequest(new FreshEscapedName(library, GraphFileResource.getInstance(graph).HasSystemResource, name));
	}

	@Override
	public Resource createRoot(WriteOnlyGraph graph, Root root) throws DatabaseException {
		
		Layer0 l0 = graph.getService(Layer0.class);

		this.root = graph.newResource();
		String name = root.name;
		String newName = nameMappings.get(name);
		graph.addLiteral(this.root, l0.HasName, l0.NameOf, l0.String, newName, Bindings.STRING);
		return this.root;

	}
	
	public Resource getRoot() {
		return root;
	}
	
	public void attachRoot(WriteGraph graph) throws DatabaseException {
		GraphFileResource gf = GraphFileResource.getInstance(graph);
		if (graph.isInstanceOf(library, gf.Folder)) {
			if (graph.isInstanceOf(root, gf.Folder)) {
				graph.claim(library, gf.HasFolder, root);
				return;
			}  else if (graph.isInstanceOf(root, gf.File)) {
				graph.claim(library, gf.HasFile, root);
				return;
			}
		}
		throw new DatabaseException("Unknown type, cannot attach copied resource " + root);
	}

}
