/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.structural2.utils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.Statement;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.CommentMetadata;
import org.simantics.db.common.request.ObjectsWithType;
import org.simantics.db.common.request.PossibleTypedParent;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.exception.MissingVariableException;
import org.simantics.db.layer0.request.InstantiateRequest;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.layer0.Layer0;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.structural2.Functions.StructuralOverrideData;
import org.simantics.structural2.internal.queries.ConnectedTo;
import org.simantics.structural2.internal.queries.RelatedConnections;
import org.simantics.structural2.internal.queries.RelatedConnectionsOfConnectionJoin;
import org.simantics.structural2.queries.Terminal;
import org.simantics.structural2.variables.Connection;
import org.simantics.utils.datastructures.Pair;

import gnu.trove.set.hash.THashSet;

/**
 * A utility class for manipulating structural models.
 * 
 * @author Hannu Niemist&ouml;
 */
public class StructuralUtils {
    
    public static enum StructuralComponentClass {

        PRIMITIVE,REPLACEABLE,DEFINED,PROCEDURAL;

        public static StructuralComponentClass get(ReadGraph graph, Resource componentType) throws DatabaseException {
            StructuralResource2 STR = StructuralResource2.getInstance(graph);
            Set<Resource> types = graph.getTypes(componentType);
            if(types.contains(STR.ProceduralComponentType))
                return StructuralComponentClass.PROCEDURAL;
            else if(graph.hasStatement(componentType, STR.IsDefinedBy))
                return StructuralComponentClass.DEFINED;
            else if(types.contains(STR.ReplaceableDefinedComponentType))
                return StructuralComponentClass.REPLACEABLE;
            else
                return StructuralComponentClass.PRIMITIVE;
        }

    }

    public static Collection<Resource> getConnectionRelations(ReadGraph graph, Resource componentType) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        StructuralResource2 STR = StructuralResource2.getInstance(graph);
        return graph.syncRequest(new ObjectsWithType(componentType, L0.DomainOf, STR.ConnectionRelation));
    }

    public static Collection<Resource> getPropertyRelations(ReadGraph graph, Resource componentType) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        ArrayList<Resource> result = new ArrayList<Resource>();
        for(Resource relation : graph.getObjects(componentType, L0.DomainOf))
            if(graph.isSubrelationOf(relation, L0.HasProperty))
                result.add(relation);
        return result;
    }
    
    public static boolean isDomainOfRelation(ReadGraph graph,
            Resource componentType, Resource connectionRelation) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        for(Resource domain : graph.getObjects(connectionRelation, L0.HasDomain))
            if(graph.isInheritedFrom(componentType, domain))
                return true;
        return false;
    }
    
    public static void addConnectionPoint(WriteGraph g, Resource componentType,
            Resource connectionPoint) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(g);
        g.claim(connectionPoint, L0.HasDomain, componentType);
    }
    
    public static Resource createConnectionPoint(WriteGraph g, Resource componentType, Resource copy) throws DatabaseException {
        String proposition = NameUtils.getSafeName(g, copy);
        String newName = NameUtils.findFreshName(g, proposition, componentType);
        return createConnectionPoint(g, componentType, copy, newName);
    }
    
    public static Resource createConnectionPoint(WriteGraph g, Resource componentType, Resource copy, String name) throws DatabaseException {
        return createConnectionPointP(g, componentType, copy, name).first;
    }
    
    public static Pair<Resource,Resource> createConnectionPointP(WriteGraph g, Resource componentType, Resource copy, String name) throws DatabaseException {
        
        Layer0 L0 = Layer0.getInstance(g);
        StructuralResource2 STR = StructuralResource2.getInstance(g);

        // Create the connection point
        Resource connectionPoint = g.newResource();
        Resource connectionPointInv = g.newResource();

        for (Resource superrelation : g.getObjects(copy, L0.SubrelationOf)) {
            g.claim(connectionPoint, L0.SubrelationOf, null, superrelation);
            Resource inverse = g.getPossibleInverse(superrelation);
            if (inverse != null)
                g.claim(connectionPointInv, L0.SubrelationOf, null, inverse);
        }
        for (Resource type : g.getObjects(copy, L0.InstanceOf)) {
            g.claim(connectionPoint, L0.InstanceOf, null, type);
        }
        for (Resource attachment : g.getObjects(copy, STR.HasAttachmentRelation)) {
            g.claim(connectionPoint, STR.HasAttachmentRelation, attachment);
        }
        for (Statement stm : g.getStatements(copy, STR.AllowsConnectionType)) {
            if (!stm.isAsserted(copy)) {
                g.claim(connectionPoint, stm.getPredicate(), stm.getObject());
            }
        }

        g.claim(connectionPoint, L0.InverseOf, connectionPointInv);
        g.claimLiteral(connectionPoint, L0.HasName, name, Bindings.STRING);
        g.claim(connectionPoint, L0.ConsistsOf, connectionPointInv);
        g.claimLiteral(connectionPointInv, L0.HasName, "Inverse", Bindings.STRING);

        StructuralUtils.addConnectionPoint(g, componentType, connectionPoint);

        g.claim(componentType, L0.ConsistsOf, connectionPoint);

        return Pair.make(connectionPoint, connectionPointInv);
        
    }

    /**
     * Creates a new component and the templates associated with it.
     * This method does not check whether there already exists 
     * a component with the same name and parent.
     * @param g
     * @param parent The parent composite of the new component.
     * @param name The name of the new component.
     * @param componentType The type of the new component.
     * @return
     * @throws DatabaseException
     */
    public static Resource newComponent(WriteGraph g, Resource parent, String name, Resource componentType) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(g);
        
        HashMap<String,Object> parameters = new HashMap<String,Object>();
        parameters.put("parent", parent);

        InstantiateRequest ir = new InstantiateRequest(componentType, parameters);
        Resource component = ir.perform(g);        
        g.claim(component, L0.HasName, Layer0Utils.literal(g, name));
        g.claim(component, L0.HasLabel, Layer0Utils.literal(g, ""));
        g.claim(parent, L0.ConsistsOf, component);
        Layer0Utils.claimNewIdentifier(g, component, true);
        // Add comment to change set.
        CommentMetadata cm = g.getMetadata(CommentMetadata.class);
        g.addMetadata(cm.add("Created component " + component));
        return component;
    }
    
    /**
     * Returns all child components of a composite.
     */
    public static Collection<Resource> getChildComponents(ReadGraph g, Resource parent) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(g);
        StructuralResource2 STR = StructuralResource2.getInstance(g);
        
        Collection<Resource> allChildren = g.getObjects(parent, L0.ConsistsOf);
        ArrayList<Resource> result = new ArrayList<Resource>(allChildren.size());
        for(Resource child : allChildren)
            // Composite may contain other things than components, therefore we must do checking
            if(g.isInstanceOf(child, STR.Component))
                result.add(child);
        return result;
    }

    /**
     * Returns the component type of the given component or null if the 
     * parameter is not a component.
     */
    @Deprecated
    public static Resource getComponentType(ReadGraph g, Resource component) throws DatabaseException {
        StructuralResource2 STR = StructuralResource2.getInstance(g);
        return g.getPossibleType(component, STR.Component);
    }
    
    /**
     * Returns the definitions of the component type or null, if the component
     * type does not have a definition.
     */
    public static Resource getComponentTypeDefinition(ReadGraph g, Resource componentType) throws DatabaseException {
        StructuralResource2 STR = StructuralResource2.getInstance(g);
        return g.getPossibleObject(componentType, STR.IsDefinedBy);
    }
    
    /**
     * Returns all (component,connectionRelation) pairs that are connected
     * to the given component and connectionRelation.
     * @param component
     * @param bindingRelation
     * @return
     */
    public static Set<Terminal> getRelatedTerminals(RequestProcessor g, Resource component, Resource connectionRelation) throws DatabaseException {
        return g.syncRequest(new ConnectedTo(component, connectionRelation));
    }
    
    /**
     * Returns all connections that are reachable from the given connection
     * with IsJoinedBy-relation including the given connection itself. 
     */ 
    public static Set<Resource> getRelatedConnections(RequestProcessor g, Resource connection) throws DatabaseException {
        return g.syncRequest(new RelatedConnections(connection));
    }
    
    /**
     * Returns all connections that are reachable from the given connection join.     * 
     */ 
    public static Set<Resource> getRelatedConnectionsOfConnectionJoin(RequestProcessor g, Resource connectionJoin) throws DatabaseException {
        return g.syncRequest(new RelatedConnectionsOfConnectionJoin(connectionJoin));
    }
    
    /**
     * Returns all connections that are reachable from the given connection join
     * without visiting the resources in the collection excludedConnections.
     */
    public static Set<Resource> getRelatedConnectionsOfConnectionJoin(RequestProcessor g, Resource connectionJoin, Collection<Resource> excludedConnections) throws DatabaseException {
        return g.syncRequest(new RelatedConnectionsOfConnectionJoin(connectionJoin, excludedConnections));
    }
    
    /**
     * Returns true if the given composite is a parent of some/all components 
     * where the connection is attached to.
     */
    public static boolean isConnectionInComposite(ReadGraph g, Resource connection, Resource composite) throws DatabaseException {
        StructuralResource2 STR = StructuralResource2.getInstance(g);
        Layer0 L0 = Layer0.getInstance(g);
        Collection<Resource> connects = g.getObjects(connection, STR.Connects);
        if (!connects.isEmpty()) {
            for(Resource component : connects)
                for(Resource parent : g.getObjects(component, L0.PartOf))
                    if(parent.equals(composite))
                        return true;
        } else {
            Set<Resource> joinedComposites = null;
            for(Resource join : g.getObjects(connection, STR.IsJoinedBy)) {
                Collection<Resource> joined = g.getObjects(join, STR.JoinsComposite);
                if (joinedComposites == null) {
                    joinedComposites = new THashSet<Resource>(joined);
                } else {
                    joinedComposites.retainAll(joined);
                }
            }
            if (joinedComposites != null && joinedComposites.size() == 1) {
                return joinedComposites.contains(composite);
            }
        }
        return false;
    }
    
    public static Variable getConnectionPoint(ReadGraph graph, Variable component, Resource relation) throws DatabaseException {
    	Layer0 L0 = Layer0.getInstance(graph);
    	String relationName = graph.getRelatedValue(relation, L0.HasName, Bindings.STRING);
    	return component.getProperty(graph, relationName);
    }

    public static Variable getPossibleConnectionPoint(ReadGraph graph, Variable component, Resource relation) throws DatabaseException {
    	try {
    		return getConnectionPoint(graph, component, relation);
    	} catch (MissingVariableException e) {
    		return null;
    	}
    }

    public static Variable getPossibleConnectionTo(ReadGraph graph, Variable component, Resource relation) throws DatabaseException {
    	Variable property = component.getPossibleProperty(graph, relation);
    	if(property == null) return null;
    	Connection conn = property.getPossibleValue(graph);
    	if(conn == null) return null;
    	Collection<Variable> cps = conn.getConnectionPoints(graph, null);
    	if(cps.size() == 2) {
    		for(Variable var : cps) {
    			if(property.equals(var)) continue;
    			return var;
    		}
    	}
    	return null;
    }

    public static boolean isImmutable(ReadGraph graph, Resource r) throws DatabaseException {
        StructuralResource2 STR = StructuralResource2.getInstance(graph);
        Resource uc = graph.syncRequest(new PossibleTypedParent(r, STR.ComponentType));
        return graph.isImmutable(r)
                // Anything under a published or locked user component is published as well
                || (uc != null && (Layer0Utils.isPublished(graph, uc)
                         || graph.hasStatement(uc, STR.ComponentType_Locked)))
                // Anything under a published container (shared library) is published as well
                || Layer0Utils.isContainerPublished(graph, r)
                ;
    }
    
    public static List<Variable> structuralConnectionConnectionPoints(ReadGraph graph, Connection conn, Resource relationType) throws DatabaseException {
        return new ArrayList<Variable>(conn.getConnectionPoints(graph, relationType));
    }
    
    public static Resource structuralTypeResource(ReadGraph graph, Variable component, Resource baseType) throws DatabaseException {
        StructuralOverrideData od = StructuralOverrideData.compute(graph, component);
        if (od != null)
            return od.type();
        return null;
    }
    
    public static Resource getComponentType(ReadGraph graph, Variable configuration, Resource component) throws DatabaseException {
        Variable componentVariable = configuration.browse(graph, component);
        return componentVariable.getType(graph);
    }

    public static Resource getPossibleComponentType(ReadGraph graph, Variable configuration, Resource component) throws DatabaseException {
        Variable componentVariable = configuration.browsePossible(graph, component);
        if(componentVariable == null) return null;
        StructuralResource2 STR = StructuralResource2.getInstance(graph);
        return componentVariable.getPossibleType(graph, STR.Component);
    }


}
