/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.adapter;

import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.G2DUtils;
import org.simantics.diagram.elements.ElementPropertySetter;
import org.simantics.diagram.elements.MonitorClass;
import org.simantics.diagram.elements.ResizeRectangularSceneGraph;
import org.simantics.diagram.elements.TextElementHandler;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.diagram.synchronization.CompositeHintSynchronizer;
import org.simantics.diagram.synchronization.IHintSynchronizer;
import org.simantics.diagram.synchronization.SynchronizationHints;
import org.simantics.diagram.synchronization.graph.DiagramGraphUtil;
import org.simantics.diagram.synchronization.graph.TextSynchronizer;
import org.simantics.diagram.synchronization.graph.TransformSynchronizer;
import org.simantics.g2d.canvas.ICanvasContext;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.ElementHints;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.Text;
import org.simantics.g2d.element.handler.impl.DefaultTransform;
import org.simantics.g2d.element.handler.impl.SimpleElementLayers;
import org.simantics.g2d.element.handler.impl.StaticObjectAdapter;
import org.simantics.g2d.element.handler.impl.StaticSymbolImpl;
import org.simantics.g2d.element.handler.impl.TextColorImpl;
import org.simantics.g2d.element.handler.impl.TextFontImpl;
import org.simantics.g2d.image.DefaultImages;
import org.simantics.g2d.image.Image;
import org.simantics.g2d.svg.SVGImage;
import org.simantics.g2d.utils.Alignment;


/**
 * Element(Class)Factory for TextElements
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 *
 */
public class TextElementClassFactory extends SyncElementFactory {

    public static final ElementFactory INSTANCE = new TextElementClassFactory();

    public static final IHintSynchronizer TEXT_SYNCHRONIZER = new CompositeHintSynchronizer(TextSynchronizer.INSTANCE, TransformSynchronizer.INSTANCE);

    public static final ElementPropertySetter RESIZE_PROPERTY_SETTER =
            new ElementPropertySetter(ResizeRectangularSceneGraph.KEY_SG_NODE);
    
    @Override
    public ElementClass create(ReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource elementType) throws DatabaseException {
        String id = "TextElement: " + NameUtils.getSafeName(graph, elementType);
        G2DResource g2d = G2DResource.getInstance(graph);
        String svgDoc = graph.getPossibleRelatedValue(elementType, g2d.HasSVGDocument);
        Image image = null;
        if (svgDoc != null)
            image = new SVGImage(id+".svg", svgDoc);
        else
            image = DefaultImages.ERROR_DECORATOR.get();
        return ElementClass.compile(
                Text.INSTANCE,
                TextColorImpl.BLACK,
                TextFontImpl.DEFAULT,
                new StaticObjectAdapter(elementType),
                DefaultTransform.INSTANCE,
                SimpleElementLayers.INSTANCE,
                TextElementHandler.INSTANCE,
                ResizeRectangularSceneGraph.INSTANCE,
                RESIZE_PROPERTY_SETTER,                
                new StaticSymbolImpl(image)
        )
        .setId(id)
        ;
    }

    @Override
    public void load(ReadGraph graph, ICanvasContext canvas, IDiagram diagram, Resource element, IElement e) throws DatabaseException {
        G2DResource G2D = G2DResource.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);

        String text = (String) graph.getPossibleRelatedValue(element, DIA.HasText);
        if (text == null)
        	text = ElementUtils.getText(e);
        if (text == null)
            text = "[empty]";
        ElementUtils.setText(e, text);

        if (graph.isInstanceOf(element, DIA.FontProvider)) {
            Resource fontResource = graph.getPossibleObject(element, G2D.HasFont);
            if (fontResource != null)
                ElementUtils.setTextFont(e, G2DUtils.getFont(graph, fontResource));
        }
        if (graph.isInstanceOf(element, DIA.ColorProvider)) {
            Resource colorResource = graph.getPossibleObject(element, G2D.HasColor);
            if (colorResource != null)
                ElementUtils.setTextColor(e, G2DUtils.getColor(graph, colorResource));
        }

        // Load alignments
        Resource hAlign = graph.getPossibleObject(element, G2D.HasHorizontalAlignment);
        Resource vAlign = graph.getPossibleObject(element, G2D.HasVerticalAlignment);
        if (hAlign != null)
            e.setHint(ElementHints.KEY_HORIZONTAL_ALIGN, DiagramGraphUtil.toAlignment(hAlign, G2D, Alignment.LEADING));
        if (vAlign != null)
            e.setHint(ElementHints.KEY_VERTICAL_ALIGN, DiagramGraphUtil.toVerticalAlignment(vAlign, G2D, Alignment.BASELINE));

        // Load border
        final Float borderWidth = graph.getPossibleRelatedValue(element, G2D.HasStrokeWidth);
        if (borderWidth != null)
            e.setHint(MonitorClass.KEY_BORDER_WIDTH, borderWidth.doubleValue());

        // Load transform
        AffineTransform at = DiagramGraphUtil.getAffineTransform(graph, element);
        ElementUtils.setTransform(e, at);

        // This synchronizes only text and transformation (not font and color)
        e.setHint(SynchronizationHints.HINT_SYNCHRONIZER, TEXT_SYNCHRONIZER);
        
        double bounds[] = DiagramGraphUtil.getPossibleRelatedDoubleArray(graph, element, G2D.HasBounds);
        if (bounds != null) {
            e.setHint(ElementHints.KEY_BOUNDS, new Rectangle2D.Double(bounds[0], bounds[1], bounds[2], bounds[3]));
        }
    }

}
