/*******************************************************************************
 * Copyright (c) 2007, 2011 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.profile.view;

import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.IPartService;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.EditorPart;
import org.simantics.Simantics;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.common.ErrorLogger;
import org.simantics.browsing.ui.graph.impl.InputSourceListener;
import org.simantics.browsing.ui.graph.impl.ObservableInputSource;
import org.simantics.browsing.ui.graph.impl.WorkbenchSessionContextInputSource;
import org.simantics.db.Disposable;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.procedure.Procedure;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.utils.ObjectUtils;
import org.simantics.utils.ui.SWTUtils;

/**
 * @author Tuukka Lehtonen
 */
public class ActiveRuntimeDiagramInputSource implements WorkbenchSessionContextInputSource, ObservableInputSource, IPartListener2, Disposable {

    protected Display             display;
    protected IWorkbenchPart      ownerPart;
    protected IPartService        service;
    protected IEditorPart         activeEditor;
    protected Resource            activeRuntimeDiagram;
    protected InputSourceListener listener;
    protected boolean             ownerIsVisible = true;
    protected Resource            lastInputResource;

    @Override
    public void init(IWorkbenchSite site, IWorkbenchPart part) {
        this.display = site.getShell().getDisplay();
        this.ownerPart = part;
        attachToWorkbench();
    }

    /**
     * @thread SWT
     */
    protected void attachToWorkbench() {
        IWorkbenchWindow window = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
        if (window == null)
            return;
        service = window.getPartService();
        service.addPartListener(this);

        IWorkbenchPage activePage = window.getActivePage();
        if (activePage == null)
            return;
        
        IEditorPart activeEditor = activePage.getActiveEditor();
        if (activeEditor != null)
            editorActivated(activeEditor);
    }

    /**
     * @thread SWT
     */
    @Override
    public void dispose() {
        if (service != null)
            service.removePartListener(this);
        service = null;
        activeEditor = null;
        activeRuntimeDiagram = null;
        listener = null;
        ownerPart = null;
        display = null;
    }

    protected void fireIfInputChanged(Object oldSelection, Object newSelection) {
        InputSourceListener l = listener;
        if (l != null)
            if (!ObjectUtils.objectEquals(oldSelection, newSelection))
                l.inputChanged(this);
    }

    @Override
    public Object get(ISessionContext ctx) {
        return activeRuntimeDiagram != null ? activeRuntimeDiagram : GraphExplorer.EMPTY_INPUT;
    }
    
    @Override
    public IWorkbenchPart getProvider() {
    	return activeEditor;
    }

    @Override
    public void setListener(InputSourceListener listener) {
        this.listener = listener;
    }

    private static class InputTransformation extends UniqueRead<Resource> {

        private Resource resource;
        private Resource defaultResult;

        public InputTransformation(Resource input, Resource defaultResult) {
            this.resource = input;
            this.defaultResult = defaultResult;
        }

        @Override
        public Resource perform(ReadGraph graph) throws DatabaseException {
            if (graph.isInstanceOf(resource, DiagramResource.getInstance(graph).RuntimeDiagram))
                return resource;
            if (!graph.hasStatement(resource))
                return null;

//            if (defaultResult != null && !graph.hasStatement(defaultResult))
//                return null;

            return defaultResult;
        }

    }

    class InputProcedure implements Procedure<Resource> {
        @Override
        public void execute(Resource result) {
            Display d = display;
            if (d != null)
                SWTUtils.asyncExec(d, () -> changeInput(result));
        }
        @Override
        public void exception(Throwable t) {
            ErrorLogger.defaultLogError(t);
        }
    }

    private InputProcedure inputProcedure = new InputProcedure();

    protected void testAndChangeInput(Resource resource) {
        Session session = Simantics.peekSession();
        if (session != null && resource != null) {
            session.asyncRequest(
                    new InputTransformation(resource, activeRuntimeDiagram),
                    inputProcedure);
        } else {
            changeInput(null);
        }
    }

    protected void editorActivated(IEditorPart part) {
        Resource resource = part.getAdapter(Resource.class); 
        lastInputResource = resource;
        if (ownerIsVisible) {
            testAndChangeInput(resource);
        }
    }

    private boolean allEditorsClosed(IEditorPart forWindowOfPart) {
        for (IWorkbenchPage page : forWindowOfPart.getEditorSite().getWorkbenchWindow().getPages()) {
            IEditorReference[] editors = page.getEditorReferences();
            if (editors.length > 0)
                return false;
        }
        return true;
    }

    protected void activeEditorClosed() {
        if (allEditorsClosed(activeEditor))
            changeInput(null);
    }

    private void changeInput(Resource newInput) {
        Resource oldInput = activeRuntimeDiagram;
        this.activeRuntimeDiagram = newInput;
        fireIfInputChanged(oldInput, newInput != null ? newInput : GraphExplorer.EMPTY_INPUT);
    }

    @Override
    public void partActivated(IWorkbenchPartReference partRef) {
        //System.out.println("partActivated1: " + partRef);
        if (partRef instanceof IEditorReference) {
            IEditorPart part = ((IEditorReference) partRef).getEditor(false);
            //System.out.println("partActivated2: " + part);
            if (part instanceof EditorPart) {
                editorActivated((IEditorPart) part);
            }
        }
    }

    @Override
    public void partBroughtToTop(IWorkbenchPartReference partRef) {
        //System.out.println("partBroughtToTop: " + partRef);
    }

    @Override
    public void partClosed(IWorkbenchPartReference partRef) {
        //System.out.println("partClosed1: " + partRef);
        if (partRef instanceof IEditorReference) {
            IEditorPart part = ((IEditorReference) partRef).getEditor(false);
            //System.out.println("partClosed2: " + part);
            if (part == activeEditor) {
                activeEditorClosed();
            }
        }
    }

    @Override
    public void partDeactivated(IWorkbenchPartReference partRef) {
        //System.out.println("partDeactivated: " + partRef);
    }

    @Override
    public void partOpened(IWorkbenchPartReference partRef) {
        //System.out.println("partOpened: " + partRef);
    }

    @Override
    public void partInputChanged(IWorkbenchPartReference partRef) {
        //System.out.println("partInputChanged: " + partRef);
    }

    @Override
    public void partHidden(IWorkbenchPartReference partRef) {
        //System.out.println("partHidden: " + partRef);
        IWorkbenchPart part = partRef.getPart(false);
        if (ownerPart != null && ownerPart.equals(part)) {
            ownerIsVisible = false;
        }
    }

    @Override
    public void partVisible(IWorkbenchPartReference partRef) {
        //System.out.println("partVisible: " + partRef);
        IWorkbenchPart part = partRef.getPart(false);
        if (ownerPart != null && ownerPart.equals(part)) {
            ownerIsVisible = true;
            testAndChangeInput(lastInputResource);
        }
    }

}
