/*******************************************************************************
 * Copyright (c) 2007, 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.views.swt;

import org.eclipse.jface.viewers.ISelection;
import org.simantics.Simantics;
import org.simantics.browsing.ui.swt.widgets.impl.Widget;
import org.simantics.browsing.ui.swt.widgets.impl.WidgetSupportImpl;
import org.simantics.databoard.Bindings;
import org.simantics.db.AsyncReadGraph;
import org.simantics.db.Disposable;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.db.management.ISessionContext;
import org.simantics.db.procedure.AsyncListener;
import org.simantics.db.procedure.Listener;
import org.simantics.scenegraph.ontology.ScenegraphResources;
import org.simantics.ui.selection.AnyResource;
import org.simantics.ui.selection.AnyVariable;
import org.simantics.ui.selection.WorkbenchSelectionElement;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.ui.ISelectionUtils;

/**
 * @author Antti Villberg
 */
public class ModelledSupport extends WidgetSupportImpl implements Disposable {

    private final ModelledView view;
    private InputListener listener;
    private boolean disposed = false;
    private Pair<Variable, Resource> previous = null;

    /**
     * @param modelledConfigurationComposite
     */
    public ModelledSupport(ModelledView view) {
        this.view = view;
    }

    @Override
    public void dispose() {
    	
        disposed = true;
        
        if (listener != null)
            listener.dispose();
        
    }

    class InputListener implements AsyncListener<Pair<Variable, Resource>> {

        private boolean disposed = false;

        @Override
        public boolean isDisposed() {
            return this.disposed || ModelledSupport.this.disposed;
        }

        public void dispose() {
            this.disposed = true;
        }

        @Override
        public String toString() {
            return "InputListener@ModelledSupport@" + view; 
        }

        @Override
        public void execute(AsyncReadGraph graph, final Pair<Variable, Resource> result) {

            if(previous != null && previous.equals(result)) return;
            previous = result;

            graph.getSession().async(new WriteRequest(Simantics.getSession().getService(VirtualGraph.class)) {

                @Override
                public void perform(WriteGraph graph) throws DatabaseException {

                    ScenegraphResources SG = ScenegraphResources.getInstance(graph);

                    Resource runtime = view.runtime;

                    if(result.first != null) {
                        String uri = result.first.getURI(graph);
                        graph.claimLiteral(runtime, SG.Runtime_HasVariable, uri, Bindings.STRING);
                    } else {
                        graph.deny(runtime, SG.Runtime_HasVariable);
                    }

                    if(result.second != null) {
                        graph.deny(runtime, SG.Runtime_HasResource);
                        graph.claim(runtime, SG.Runtime_HasResource, result.second);
                    } else {
                        graph.deny(runtime, SG.Runtime_HasResource);
                    }

                    view.fireInput();

                }

            });

        }

        @Override
        public void exception(AsyncReadGraph graph, Throwable throwable) {
            Logger.defaultLogError(throwable);
        }

    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    @Override
    public void fireInput(final ISessionContext context, final Object input) {

        lastInput = input;
        lastContext = context;

        if (listener != null)
            listener.dispose();

        if (context == null)
            return;

        Session session = context.peekSession();
        if (session == null)
            return;

        listener = new InputListener();

        if(view.runtime != null) {

            session.async(new UniqueRead<Pair<Variable, Resource>>() {

                private Variable extractVariable(ReadGraph graph, ISelection selection) throws DatabaseException {

                    WorkbenchSelectionElement single = ISelectionUtils.filterSingleSelection(selection, WorkbenchSelectionElement.class);
                    if(single != null) return single.getContent(new AnyVariable(graph));
                    return ISelectionUtils.filterSingleSelection((ISelection)lastInput, Variable.class);

                }

                private Resource extractResource(ReadGraph graph, ISelection selection) throws DatabaseException {

                    WorkbenchSelectionElement single = ISelectionUtils.filterSingleSelection(selection, WorkbenchSelectionElement.class);
                    if(single != null) return single.getContent(new AnyResource(graph));
                    return ISelectionUtils.filterSingleSelection((ISelection)lastInput, Resource.class);

                }

                @Override
                public Pair<Variable, Resource> perform(ReadGraph graph) throws DatabaseException {

                    Variable resultVariable = null;
                    Resource resultResource = null; 

                    if(lastInput instanceof ISelection) {
                        Variable var = extractVariable(graph, (ISelection)lastInput);
                        if(var != null) resultVariable = var;
                        Resource res = extractResource(graph, (ISelection)lastInput);
                        if(res != null) resultResource = res;
                    } else if (lastInput instanceof Resource) {
                        resultResource = (Resource)lastInput;
                    }

                    return Pair.make(resultVariable, resultResource);

                }

            }, listener);

        } else {

            for(Widget widget : widgets) {
                widget.setInput(context, input);
                for (Listener listener : listeners)
                    listener.execute(input);
            }

        }
    }

}
