/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.service;

import java.util.Collection;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.ServiceLocator;
import org.simantics.db.Session;
import org.simantics.db.Statement;
import org.simantics.db.VirtualGraph;
import org.simantics.db.VirtualGraph.Persistency;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.exception.DatabaseException;


/**
 * A {@link Session} service for working with virtual graphs. Virtual graphs
 * offer a way to augment the persistent graph database with
 * memory/workspace-persistent information. Use
 * {@link ServiceLocator#getService(Class)} to retrieve this support from a
 * {@link Session}.
 * 
 * See <a href="http://dev.simantics.org/index.php/Virtual_Graphs">Simantics
 * developer wiki</a> for documentation.
 * 
 * @author Antti Villberg
 */
public interface VirtualGraphSupport {

    /**
     * Gets previously initialized or creates new memory-persistent
     * (session-transient) virtual graph by the specified identifier.
     * 
     * The contents of memory-persistent graphs are lost
     * when the session finishes.
     * 
     * @param identifier memory-persistent virtual graph identifier
     */
    VirtualGraph getMemoryPersistent(String identifier);

    /**
     * First tries to return a previously loaded workspace-persistent graph by
     * identifier. Then tries to restore the graph from disk by identifier.
     * Finally creates a new workspace-persistent graph.
     * 
     * The graph is automatically persisted to disk.
     * 
     * @param identifier workspace-persistent virtual graph identifier
     */
    VirtualGraph getWorkspacePersistent(String identifier);

    /**
     * Removes this graph from the set of active graphs. If the graph is
     * workspace persistent, the contents are deleted from disk.
     * 
     * @return <code>true</code> if the virtual graph was properly discarded
     */
    boolean discard(VirtualGraph graph);

    /**
     * Transfers the contents of this virtual graph into the persistent store
     * 
     * @return <code>true</code> if the virtual graph was properly integrated
     */
    boolean integrate(WriteOnlyGraph graph, VirtualGraph vg) throws DatabaseException;
    
    /**
     * Lists all active virtual graphs currently attached to the owner database
     * session.
     * 
     * @return list of active virtual graphs
     */
    Collection<VirtualGraph> listGraphs();

    /**
     * List all statements in the specified virtual graph.
     * 
     * @param graph the virtual graph to get statements from
     * @return statements in the specified virtual graph
     */
    Collection<Statement> listStatements(VirtualGraph graph);

    /**
     * List all resources that contain literal values in the specified virtual
     * graph.
     * 
     * @param graph the virtual graph to get resources from
     * @return resources in the specified virtual graph containing literals
     */
    Collection<Resource> listValues(VirtualGraph graph);
    
    VirtualGraph getGraph(ReadGraph graph, Resource subject, Resource predicate, Resource object) throws DatabaseException;

    VirtualGraph getGraph(ReadGraph graph, Resource subject, Resource predicate) throws DatabaseException;

    VirtualGraph getGraph(ReadGraph graph, Resource subject) throws DatabaseException;

    /**
     * A simple method for persisting the current state of all managed virtual
     * graphs where {@link VirtualGraph#getPersistency()} is
     * {@link Persistency#WORKSPACE}.
     */
    void saveAll() throws DatabaseException;

}
