/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui;


/**
 * Immutable descriptor for an IGraphExplorer Column.
 * 
 * Columns can be specified to be space-grabbing or non-space-grabbing. When a
 * column is marked as grabbing, it will receive/lose its own weight's worth of
 * screen space when the parenting control is resized in a way that allows
 * columns to shrink or grow.
 * 
 * Each column is identified by a string that must be unique among the set of
 * columns used at a time. The key must be a valid string, <code>null</code> is
 * not allowed.
 * 
 * See <code>@see org.simantics.browsing.ui.common.ColumnKeys</code> for some
 * commonly usable column keys.
 * 
 * @see ColumnFactory
 * 
 * @author Tuukka Lehtonen
 */
public final class Column {

    public static enum Align {
        LEFT,
        CENTER,
        RIGHT
    }

    public static final int UNDEFINED_CONTROL_WIDTH = -1;

    public static final int DEFAULT_CONTROL_WIDTH   = UNDEFINED_CONTROL_WIDTH;

    /**
     * Name of the column.
     * @see ColumnKeys
     */
    private final String key;

    /**
     * Text to be shown in the column.
     * @see ColumnKeys
     */
    private final String label;

    /**
     * Content alignment used by this column.
     */
    private final Align  alignment;

    /**
     * -1 means not specified. If grab == false, this describes the default
     * width, if grab == true, this describes the minimum width of the column.
     */
    private final int    width;

    /**
     * A string to show as tooltip or <code>null</code>.
     */
    private final String tooltip;

    /**
     * Indicates whether this column participates in grabbing extra widget space
     * that is freed when columns are resized.
     */
    private final boolean grab;

    /**
     * Weight describes how much a space-grabbing column will grab or release of
     * screen space that is gained or lost when controls are resized. The
     * percentage of space given to a column is defined by the formula:
     * <code>100*weight / sum(all column weights)</code>
     */
    private final int weight;

    public Column(String key) {
        this(key, key, Align.LEFT);
    }

    public Column(String key, String label, Align alignment) {
        this(key, label, alignment, DEFAULT_CONTROL_WIDTH);
    }

    public Column(String key, String label, Align alignment, int width) {
        this(key, label, alignment, width, null);
    }

    public Column(String key, Align alignment, int width) {
        this(key, key, alignment, width, null);
    }

    public Column(String key, Align alignment, int width, String tooltip) {
        this(key, key, alignment, width, tooltip);
    }

    public Column(String key, Align alignment, int width, String tooltip, boolean grab) {
        this(key, key, alignment, width, tooltip, grab);
    }

    public Column(String key, Align alignment, int width, String tooltip, boolean grab, int weight) {
        this(key, key, alignment, width, tooltip, grab, weight);
    }

    public Column(String key, String label, Align alignment, int width, String tooltip) {
        this(key, label, alignment, width, tooltip, false);
    }

    public Column(String key, String label, Align alignment, int width, String tooltip, boolean grab) {
        this(key, label, alignment, width, tooltip, grab, 1);
    }

    public Column(String key, String label, Align alignment, int width, String tooltip, boolean grab, int weight) {
        if (alignment == null)
            throw new IllegalArgumentException("null alignment"); //$NON-NLS-1$
        if (key == null)
            throw new IllegalArgumentException("null key"); //$NON-NLS-1$

        this.key = key;
        this.label = label;
        this.alignment = alignment;
        this.width = width;
        this.tooltip = tooltip;
        this.grab = grab;
        this.weight = weight;
    }

    public String getKey() {
        return key;
    }

    public String getLabel() {
        return label;
    }

    public Align getAlignment() {
        return alignment;
    }

    public int getWidth() {
        return width;
    }

    public String getTooltip() {
        return tooltip;
    }

    public boolean hasGrab() {
        return grab;
    }

    public int getWeight() {
        return weight;
    }

    public Column withKey(String key) {
        return new Column(key, this.label, this.alignment, this.width, this.tooltip, this.grab, this.weight);
    }

    public Column withLabel(String label) {
        return new Column(this.key, label, this.alignment, this.width, this.tooltip, this.grab, this.weight);
    }

    public Column withAlignment(Align alignment) {
        return new Column(this.key, this.label, alignment, this.width, this.tooltip, this.grab, this.weight);
    }

    public Column withWidth(int width) {
        return new Column(this.key, this.label, this.alignment, width, this.tooltip, this.grab, this.weight);
    }

    public Column withTooltip(String tooltip) {
        return new Column(this.key, this.label, this.alignment, this.width, tooltip, this.grab, this.weight);
    }

    public Column withGrab(boolean grab) {
        return new Column(this.key, this.label, this.alignment, this.width, this.tooltip, grab, this.weight);
    }

    public Column withWeight(int weight) {
        return new Column(this.key, this.label, this.alignment, this.width, this.tooltip, this.grab, weight);
    }
    
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + alignment.hashCode();
        result = prime * result + (grab ? 1231 : 1237);
        result = prime * result + ((key == null) ? 0 : key.hashCode());
        result = prime * result + ((label == null) ? 0 : label.hashCode());
        result = prime * result + ((tooltip == null) ? 0 : tooltip.hashCode());
        result = prime * result + width;
        result = prime * result + weight;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        Column other = (Column) obj;
        if (alignment != other.alignment)
            return false;
        if (grab != other.grab)
            return false;
        if (key == null) {
            if (other.key != null)
                return false;
        } else if (!key.equals(other.key))
            return false;
        if (label == null) {
            if (other.label != null)
                return false;
        } else if (!label.equals(other.label))
            return false;
        if (tooltip == null) {
            if (other.tooltip != null)
                return false;
        } else if (!tooltip.equals(other.tooltip))
            return false;
        if (width != other.width)
            return false;
        if (weight != other.weight)
            return false;
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Column[key="); //$NON-NLS-1$
        sb.append(key);
        if (!key.equals(label))
            sb.append(", label="); //$NON-NLS-1$
        sb.append(label);
        sb.append(", align="); //$NON-NLS-1$
        sb.append(alignment);
        sb.append(", width="); //$NON-NLS-1$
        sb.append(width);
        if (!label.equals(tooltip)) {
            sb.append(", tooltip="); //$NON-NLS-1$
            sb.append(tooltip);
        }
        sb.append(", grab="); //$NON-NLS-1$
        sb.append(grab);
        sb.append(", weight="); //$NON-NLS-1$
        sb.append(weight);
        sb.append("]"); //$NON-NLS-1$
        return sb.toString();
    }

}
