/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2015 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * http://glassfish.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at packager/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 * This file incorporates work covered by the following copyright and
 * permission notice:
 *
 * Copyright 2010-2013 Coda Hale and Yammer, Inc., 2014-2015 Dropwizard Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.glassfish.jersey.server.internal.monitoring;

import java.util.concurrent.TimeUnit;

/**
 * A statistically representative reservoir of a data stream in time.
 * <p/>
 * Compared to Dropwizard Reservoir, this interface adds a possibility to work with data that is associated with a specific
 * time. It may not be possible; however, to obtain a snapshot or size at some moment in past due to performance optimizations.
 *
 * @author Stepan Vavra (stepan.vavra at oracle.com)
 * @see <a href="https://github.com/dropwizard/metrics">https://github.com/dropwizard/metrics</a>
 */
interface TimeReservoir<V> {

    /**
     * Returns the number of values recorded at given time or newer. It may not be supported to return a size in past due to
     * performance optimizations.
     *
     * @param time     The time to get the size for
     * @param timeUnit Time unit of the provided time
     * @return the number of values recorded for given time or newer
     */
    int size(long time, TimeUnit timeUnit);

    /**
     * Adds a new recorded value to the reservoir bound to a given time.
     *
     * @param value    a new recorded value
     * @param time     The time the recorded value occurred at
     * @param timeUnit Time unit of the provided time
     */
    void update(V value, long time, TimeUnit timeUnit);

    /**
     * Returns a snapshot of the reservoir's values at given time or newer. It may not be supported to return a snapshot in past
     * due to performance optimizations.
     *
     * @param time     The time for which to get the snapshot
     * @param timeUnit Time unit of the provided time
     * @return a snapshot of the reservoir's values for given time or newer
     */
    UniformTimeSnapshot getSnapshot(long time, TimeUnit timeUnit);

    /**
     * The time interval this reservoir stores data of.
     *
     * @param timeUnit The time unit in which to get the interval
     * @return The time interval of this time reservoir
     */
    long interval(TimeUnit timeUnit);
}
