/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.tests;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import junit.framework.TestCase;

import org.simantics.databoard.Bindings;
import org.simantics.databoard.binding.Binding;
import org.simantics.databoard.file.FileList;
import org.simantics.databoard.file.IFileList;

public class TestFileList2 extends TestCase {

	int length = 1024*10;

	Long[] data1array;
	List<Long> data1list;
	IFileList<Long> data1file;	
	Binding data1binding = Bindings.getBindingUnchecked(Long.class);
	
	public void setUp() 
	throws Exception {
		data1array = new Long[length];
		data1list = new ArrayList<Long>(length); 
		for (int i=0; i<length; i++)
		{
			Long s = Long.valueOf(i);
			data1array[i] = s;
			data1list.add( s );			
		}
		
		File tmp1 = File.createTempFile("Tmp", ".tmp");
		tmp1.deleteOnExit();
		
		data1file = new FileList<Long>(tmp1, Long.class); 
	}
	
	public void tearDown() 
	throws Exception 
	{
		data1file.close();
	}
	
	public void testAppend() 
	throws Exception
	{		
		// Write
		data1file.addAll(data1list);
		assertEquals(length, data1file.size());
		// Read-Verify
		int i=0;
		for (Long line : data1file)
			assertEquals(data1array[i++], line);
	}
	
	public void testClear()
	throws Exception
	{
		// Write
		data1file.addAll(data1list);
		assertFalse(data1file.isEmpty());
		assertEquals(length, data1file.size());
		// Clear all
		data1file.clear();
		assertTrue(data1file.isEmpty());
		assertEquals(0, data1file.size());
		// Check that file size has truncated
		assertTrue(data1file.getFile().length()<1024);
	}
	
	public void testRemove()
	throws Exception
	{
		data1file.addAll(data1list);
		
		// remove 10 elements, one by one
		for (int i=0; i<10; i++) {
			data1file.remove(10);
			assertEquals(length-i-1, data1file.size());
		}
		
		// check data looks sane
		for (int i=0; i<10; i++)
			assertEquals(data1array[i], data1file.get(i));
		for (int i=20; i<length; i++)
			assertEquals(data1array[i], data1file.get(i-10));
	
	}
	
	public void testSetAll()
	throws Exception
	{		
		List<Long> data2 = new ArrayList<Long>();
		for (long i=0L; i<500L; i++)
			data2.add( i + Integer.MAX_VALUE );
	
		data1file.clear();
		((FileList<Long>)data1file).setAll(data2);

		assertEquals(data1file.size(), data2.size());
		for (int i=0; i<data2.size(); i++)
			assertEquals(data2.get(i), data1file.get(i));
		
		((FileList<Long>)data1file).setAll(data1list);

		assertEquals(data1file.size(), data1list.size());
		for (int i=0; i<data1list.size(); i++)
			assertEquals(data1list.get(i), data1file.get(i));		
	}	
	
	public void testInsert()
	throws Exception
	{
		data1file.addAll(data1list);
		
		// int 10 elements, one by one
		for (int i=0; i<10; i++) {
			data1file.add(10+i, Long.valueOf(-i));
			assertEquals(length+i+1, data1file.size());
		}
		
		// check data looks sane
		for (int i=0; i<10; i++)
			assertEquals(data1array[i], data1file.get(i));
		for (int i=0; i<10; i++)
			assertEquals(Long.valueOf(-i), data1file.get(i+10));
		for (int i=20; i<length-10; i++)
			assertEquals(data1array[i+10], data1file.get(i+20));		
	}
	
	public void testInsertAll()
	throws Exception
	{
		data1file.addAll(data1list);
		data1file.addAll(0, data1list);
		for (int i=0; i<length; i++)
			assertEquals(data1array[i], data1file.get(i));
		for (int i=0; i<length; i++)
			assertEquals(data1array[i], data1file.get(i+length));
	}	
	
	public void testIterator()
	throws Exception
	{
		data1file.addAll(data1list);

		Iterator<Long> iter = data1file.iterator();
		int i=0;
		while (iter.hasNext())
			assertEquals(data1array[i++], iter.next());
		
		i=100;
		ListIterator<Long> li = data1file.listIterator(100);
		while (li.hasNext()) {
			assertEquals(li.nextIndex(), i);
			assertEquals(data1array[i++], li.next());
		}
		assertTrue(i==length);
		
		i = data1file.size();
		li = data1file.listIterator(i);
		while (li.hasPrevious()) {
			--i;
			assertEquals(li.previousIndex(), i);
			assertEquals(data1array[i], li.previous());
		}
		
		// Change to other objects 
		li = data1file.listIterator();
		while (li.hasNext()) {
			li.next();
			li.set(Long.valueOf(-li.nextIndex()));
		}
		// Verify		
		li = data1file.listIterator(0);
		while (li.hasNext()) {
			Long value = li.next();
			assertEquals(Long.valueOf(-li.nextIndex()), value);
		}		
	}
	
	public void testIndexOf() 
	throws Exception
	{
		data1file.addAll(data1list);

		assertEquals(2, data1file.indexOf(Long.valueOf(2)));
		assertEquals(99, data1file.lastIndexOf(Long.valueOf(99)));
		assertEquals(-1, data1file.indexOf(-5));
		assertEquals(-1, data1file.lastIndexOf(-5));
	}
	
	public void testContains()
	throws Exception
	{
		data1file.addAll(data1list);
		assertTrue( data1file.contains(Long.valueOf(5)) );
		assertTrue( data1file.contains(Long.valueOf(50)) );
		assertFalse( data1file.contains(Long.MAX_VALUE) );
	}
	
	public void testSubList()
	throws Exception
	{
		data1file.addAll(data1list);
		List<Long> subList = data1file.subList(10, 110);
		assertEquals(100, subList.size());
		for (int i=0; i<100; i++)
			assertEquals(data1array[i+10], subList.get(i));
	}
	
	public void testToArray()
	throws Exception
	{
		data1file.addAll(data1list);
		Long[] array = data1file.toArray(new Long[0]);
		assertTrue( Arrays.equals(array, data1array) );
	}
		
	public void testOpen()
	throws Exception
	{
		File file = data1file.getFile();
		data1file.addAll(data1list);
		data1file.close();
		data1file = new FileList<Long>(file, Long.class); 
		
		for (int i=0; i<length; i++)
			assertEquals(data1array[i], data1file.get(i));		
	}

}

