/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.workbench.editor.input;

import java.util.Arrays;

import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ParametrizedRead;
import org.simantics.db.common.request.Queries;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.request.combinations.Combinators;
import org.simantics.db.request.Read;
import org.simantics.ui.workbench.IResourceEditorInput;

/**
 * Composable database read operations designed to be used for defining
 * validation strategies for resource editor inputs through composition.
 * 
 * <p>
 * For example one validation criterion might be that the editor input or one of
 * its neighboring resources must have a proper URI.
 * 
 * @author Tuukka Lehtonen
 */
public final class InputValidationCombinators {

    /**
     * Returns a function:
     * <pre>
     *   = subject -> exists(URI(subject))
     *   | null    -> false
     * </pre>.
     */
    public static ParametrizedRead<Resource, Boolean> hasURI() {
        return HAS_URI;
    }

    /**
     * Returns a function:
     * <pre>
     *   = input -> resource(input)
     *   | null  -> null
     * </pre>.
     */
    public static ParametrizedRead<IResourceEditorInput, Resource> extractInputResource() {
        return EXTRACT_INPUT_RESOURCE;
    }

    /**
     * Returns a function:
     * <pre>
     *   = subject -> singleObject(subject, resource(relationURI))
     *   | null    -> null
     * </pre>.
     */
    public static ParametrizedRead<Resource,Resource> completeFunction(String relationURI) {
        return new CompleteFunctionURI(relationURI);
    }

    /**
     * Returns a function:
     * <pre>
     *   = subject -> possibleObject(subject, resource(relationURI))
     *   | null    -> null
     * </pre>.
     */
    public static ParametrizedRead<Resource,Resource> partialFunction(String relationURI) {
        return new PartialFunctionURI(relationURI);
    }

    /**
     * Returns a function:
     * <pre>
     *   = subject -> true if any of conditions is true for subject, false otherwise
     *   | null    -> true
     * </pre>.
     */
    @SuppressWarnings("unchecked")
    public static ParametrizedRead<Resource, Boolean> or(ParametrizedRead<Resource, Boolean> c1,
            ParametrizedRead<Resource, Boolean> c2) {
        return new FunctionOr(new ParametrizedRead[] { c1, c2 });
    }

    // ------------------------------------------------------------------------

    private static class Or extends UnaryRead<Resource, Boolean> {
        ParametrizedRead<Resource, Boolean>[] reads;
        public Or(Resource resource, ParametrizedRead<Resource, Boolean>... reads) {
            super(resource);
            this.reads = reads;
        }
        @Override
        public Boolean perform(ReadGraph graph) throws DatabaseException {
            for (ParametrizedRead<Resource, Boolean> r : reads) {
                Read<Boolean> read = r.get(parameter);
                Boolean value = graph.syncRequest( read );
                if (value)
                    return Boolean.TRUE;
            }
            return Boolean.FALSE;
        }
        @Override
        public int hashCode() {
            return super.hashCode() * 31 + Arrays.hashCode(reads);
        }
        @Override
        public boolean equals(Object object) {
            if (this == object) return true;
            else if (object == null || getClass() != object.getClass()) return false;
            Or other = (Or) object;
            return super.equals(object) && Arrays.equals(reads, other.reads);
        }
    }

    // ------------------------------------------------------------------------

    private static class FunctionOr implements ParametrizedRead<Resource, Boolean> {
        ParametrizedRead<Resource, Boolean>[] reads;
        public FunctionOr(ParametrizedRead<Resource, Boolean>... reads) {
            this.reads = reads;
        }
        @Override
        public Read<Boolean> get(Resource subject) {
            if (subject == null || reads.length == 0)
                return Combinators.constant(Boolean.TRUE);
            return new Or(subject, reads);
        }
        @Override
        public int hashCode() {
            return getClass().hashCode() + 31 * Arrays.hashCode(reads);
        }
        @Override
        public boolean equals(Object obj) {
            if(obj == this) return true;
            if(obj == null || obj.getClass() != getClass()) return false;
            FunctionOr other = (FunctionOr)obj;
            return Arrays.equals(reads, other.reads);
        }
    }

    // ------------------------------------------------------------------------

    private static class HasURI extends UnaryRead<Resource, Boolean> {
        public HasURI(Resource resource) {
            super(resource);
        }
        @Override
        public Boolean perform(ReadGraph graph) throws DatabaseException {
            if (parameter == null)
                return Boolean.FALSE;

            String uri = graph.syncRequest(Queries.possibleUri(parameter));
            //System.out.println("uri(" + parameter + "): " + uri);
            if (uri == null)
                return Boolean.FALSE;

            // FIXME: URI request will return invalid URIs, like
            // null/Configuration/MyComposite after deleting the parenting model
            // For this reason we try to reverse lookup the URI back into a
            // resource which must be equal to the original parameter resource.

            Resource reverseLookup = graph.getPossibleResource(uri);
            //System.out.println("resource(" + uri + "): " + reverseLookup);
            return parameter.equals(reverseLookup);
        }
    }

    public static Read<Boolean> hasURI(Resource resource) {
        return new HasURI(resource);
    }

    // ------------------------------------------------------------------------

    private static class ExtractResource extends UnaryRead<IResourceEditorInput, Resource> {
        public ExtractResource(IResourceEditorInput input) {
            super(input);
        }
        @Override
        public Resource perform(ReadGraph graph) throws DatabaseException {
            return parameter != null ? parameter.getResource() : null;
        }
    }

    public static Read<Resource> extractResource(IResourceEditorInput input) {
        return new ExtractResource(input);
    }

    // ------------------------------------------------------------------------

    private static class PossibleObjectURI implements Read<Resource> {
        Resource subject;
        String relationURI;
        public PossibleObjectURI(Resource subject, String relationURI) {
            this.subject = subject;
            this.relationURI = relationURI;
        }
        @Override
        public Resource perform(ReadGraph graph) throws DatabaseException {
            Resource relation = graph.getResource(relationURI);
            return graph.getPossibleObject(subject, relation);
        }
        @Override
        public int hashCode() {
            return subject.hashCode() + 31 * relationURI.hashCode();
        }
        @Override
        public boolean equals(Object object) {
            if (this == object) return true;
            else if (object == null || getClass() != object.getClass()) return false;
            PossibleObjectURI other = (PossibleObjectURI)object;
            return subject.equals(other.subject) && relationURI.equals(other.relationURI);
        }
    }

    /**
     * Returns a read request that reads an object possibly connected to the subject by the relation.
     */
    public static Read<Resource> possibleObject(Resource subject, String relationURI) {
        return new PossibleObjectURI(subject, relationURI);
    }

    // ------------------------------------------------------------------------

    private static class PartialFunctionURI implements ParametrizedRead<Resource, Resource> {
        String relationURI;
        public PartialFunctionURI(String relationURI) {
            this.relationURI = relationURI;
        }
        @Override
        public Read<Resource> get(Resource subject) {
            if (subject == null)
                return Combinators.constant(null);
            return possibleObject(subject, relationURI);
        }
        @Override
        public int hashCode() {
            return getClass().hashCode() + 31 * relationURI.hashCode();
        }
        @Override
        public boolean equals(Object obj) {
            if(obj == this) return true;
            if(obj == null || obj.getClass() != getClass()) return false;
            PartialFunctionURI other = (PartialFunctionURI)obj;
            return relationURI.equals(other.relationURI);
        }
    }

    // ------------------------------------------------------------------------

    private static class SingleObjectURI implements Read<Resource> {
        Resource subject;
        String relationURI;
        public SingleObjectURI(Resource subject, String relationURI) {
            this.subject = subject;
            this.relationURI = relationURI;
        }
        @Override
        public Resource perform(ReadGraph graph) throws DatabaseException {
            Resource relation = graph.getResource(relationURI);
            return graph.getSingleObject(subject, relation);
        }
        @Override
        public int hashCode() {
            return subject.hashCode() + 31 * relationURI.hashCode();
        }
        @Override
        public boolean equals(Object object) {
            if (this == object) return true;
            else if (object == null || getClass() != object.getClass()) return false;
            SingleObjectURI other = (SingleObjectURI) object;
            return subject.equals(other.subject) && relationURI.equals(other.relationURI);
        }
    }

    /**
     * Returns a read request that reads a single object connected to the
     * subject by the relation.
     */
    public static Read<Resource> singleObject(Resource subject, String relationURI) {
        return new SingleObjectURI(subject, relationURI);
    }

    // ------------------------------------------------------------------------

    private static class CompleteFunctionURI implements ParametrizedRead<Resource, Resource> {
        String relationURI;
        public CompleteFunctionURI(String relationURI) {
            this.relationURI = relationURI;
        }
        @Override
        public Read<Resource> get(Resource subject) {
            if (subject == null)
                return Combinators.constant(null);
            return singleObject(subject, relationURI);
        }
        @Override
        public int hashCode() {
            return getClass().hashCode() + 31 * relationURI.hashCode();
        }
        @Override
        public boolean equals(Object obj) {
            if(obj == this) return true;
            if(obj == null || obj.getClass() != getClass()) return false;
            CompleteFunctionURI other = (CompleteFunctionURI)obj;
            return relationURI.equals(other.relationURI);
        }
    }

    // ------------------------------------------------------------------------

    private static final ParametrizedRead<IResourceEditorInput, Resource> EXTRACT_INPUT_RESOURCE = new ParametrizedRead<IResourceEditorInput, Resource>() {
        @Override
        public Read<Resource> get(IResourceEditorInput input) {
            return extractResource(input);
        }
    };

    // ------------------------------------------------------------------------

    private static final ParametrizedRead<Resource, Boolean> HAS_URI = new ParametrizedRead<Resource, Boolean>() {
        @Override
        public Read<Boolean> get(Resource resource) {
            return hasURI(resource);
        }
    };

}
