/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.ui.contribution;

import java.util.Arrays;
import java.util.Comparator;

import org.eclipse.jface.action.ContributionItem;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.jface.resource.LocalResourceManager;
import org.eclipse.jface.resource.ResourceManager;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.ui.selection.WorkbenchSelectionUtils;
import org.simantics.ui.workbench.action.ResourceEditorAdapterAction;
import org.simantics.ui.workbench.editor.EditorAdapter;
import org.simantics.ui.workbench.editor.EditorRegistry;
import org.simantics.utils.strings.AlphanumComparator;


/**
 * @author Tuukka Lehtonen
 */
public class OpenWithMenuContribution extends DynamicMenuContribution {

    private static final boolean DEBUG_ADAPTERS = false;

    private ResourceManager resourceManager;

    public OpenWithMenuContribution() {
        resourceManager = new LocalResourceManager(JFaceResources.getResources());
    }

    @Override
    public void dispose() {
        if (resourceManager != null) {
            resourceManager.dispose();
            resourceManager = null;
        }
        super.dispose();
    }

    protected Object extractResource(ReadGraph graph, Object object) throws DatabaseException {
        Resource resource = WorkbenchSelectionUtils.getPossibleResourceFromSelection(graph, object);
        return resource != null ? resource : object;
    }

    @Override
    protected boolean preAcceptSelection(Object[] selection) {
        return selection.length == 1;
    }

    @Override
    protected IContributionItem[] getContributionItems(ReadGraph graph, Object[] selection) throws DatabaseException {
        final Object r = extractResource(graph, selection[0]);
        final EditorAdapter[] editorAdapters = EditorRegistry.getInstance().getAdaptersFor(graph, r);
        if (editorAdapters.length == 0)
            return NONE;

        return new IContributionItem[] {
                new ContributionItem() {
                    @Override
                    public void fill(Menu menu, int index) {
                        MenuItem openWith = new MenuItem(menu, SWT.CASCADE, index);
                        openWith.setText("Open With");
                        openWith.setEnabled(editorAdapters.length > 0);
                        Menu subMenu = new Menu(menu);
                        openWith.setMenu(subMenu);

                        if (editorAdapters.length > 0) {
                            // Sort the open with actions in descending priority order.
                            Adapter[] adapters = new Adapter[editorAdapters.length];
                            for (int i = 0; i < editorAdapters.length; i++)
                                adapters[i] = new Adapter(editorAdapters[i], r, true);
                            Arrays.sort(adapters, new Comparator<Adapter>() {
                                @Override
                                public int compare(Adapter o1, Adapter o2) {
                                    int delta = o2.getPriority() - o1.getPriority();
                                    if (delta != 0)
                                        return delta;
                                    return AlphanumComparator.CASE_INSENSITIVE_COMPARATOR.compare(o1.getText(), o2.getText());
                                }
                            });

                            for (Adapter a : adapters)
                                addMenuItem(subMenu, a, r);
                        }
                    }
                }
        };
    }

    private void addMenuItem(Menu subMenu, Adapter adapter, Object r) {
        MenuItem item = new MenuItem(subMenu, SWT.PUSH);
        String text = adapter.getText();
        if (DEBUG_ADAPTERS) {
            text = text + " (" + adapter.getAdapter().getClass().getCanonicalName() + ") [" + adapter.getPriority() + "]";
        }
        item.setText(text);
        ImageDescriptor descriptor = adapter.getImageDescriptor();
        if (descriptor != null) {
            item.setImage(resourceManager.createImage(descriptor));
        }
        item.addSelectionListener(adapter);
    }

    static class Adapter extends ResourceEditorAdapterAction implements SelectionListener {
        boolean remember;

        public Adapter(EditorAdapter adapter, Object r, boolean remember) {
            super(adapter, r);
            this.remember = remember;
        }

        @Override
        public void widgetDefaultSelected(SelectionEvent e) {
            widgetSelected(e);
        }

        @Override
        public void widgetSelected(SelectionEvent e) {
            run();
        }

        @Override
        protected void safeRun() throws Exception {
            super.safeRun();

            if (remember) {
                // Make this choice the default for the next time.
                EditorRegistry.getInstance().getMappings().put(getResource(), getAdapter());
            }
        }
    }

}
