package org.simantics.databoard.example;

import java.util.List;
import java.util.Random;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.simantics.databoard.Bindings;
import org.simantics.databoard.Datatypes;
import org.simantics.databoard.accessor.error.AccessorConstructionException;
import org.simantics.databoard.accessor.error.AccessorException;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.binding.RecordBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.forms.DataboardForm;
import org.simantics.databoard.forms.DataboardForm.Problem;
import org.simantics.databoard.type.Datatype;
import org.simantics.databoard.type.DoubleType;
import org.simantics.databoard.type.RecordType;
import org.simantics.databoard.type.StringType;
import org.simantics.databoard.type.UnionType;


public class DataboardFormExample {

	public static void main(String [] args) throws BindingException, AccessorConstructionException, AccessorException {
	    final Display display = new Display ();
	    final Shell shell = new Shell (display);
	    shell.setLayout(new FillLayout());

	    UnionType experiments = UnionType.newEnum("Result 1 - 5.11.2011 12:50",	"Result 2 - 5.11.2011 13:40", "Result 3 - 5.11.2011 15:40" ); 
	    //UnionType mergeType = UnionType.newEnum("One file", "Many files");
	    
	    
	    //mergeType = UnionType.newEnum("Merge to one file", "Separate files");

	    RecordType fileOptions = new RecordType(); 
	    fileOptions.metadata.put("style", "no-border");
	    fileOptions.addComponent("Overwrite file(s)", Datatypes.BOOLEAN);
	    fileOptions.addComponent("case1_h.pdf", DataboardForm.fileSaveDialog(".pdf", "*.pdf"));
	    fileOptions.addComponent("case1_h.aprosModel", DataboardForm.fileSaveDialog(".aprosModel", "*.aprosModel"));
	    fileOptions.addComponent("Chart.csv", DataboardForm.fileSaveDialog(".csv", "*.csv"));	    

	    RecordType dirOptions = new RecordType(); 
	    dirOptions.metadata.put("style", "no-border");
	    dirOptions.addComponent("Overwrite file(s)", Datatypes.BOOLEAN);
	    dirOptions.addComponent("Export location", DataboardForm.directoryDialog());

	    RecordType zipOptions = new RecordType(); 
	    zipOptions.metadata.put("style", "no-border");
	    zipOptions.addComponent("Overwrite file(s)", Datatypes.BOOLEAN);
	    zipOptions.addComponent("Export to .zip", DataboardForm.fileSaveDialog(".zip", "*.zip"));
	    
	    // Output format
	    RecordType outputOptions = new RecordType(); 
	    UnionType mergeType = new UnionType();
	    mergeType.metadata.put("style", "no-border");
	    mergeType.addComponent("One file", DataboardForm.fileSaveDialog("Comma Separated Value (.csv)", ".csv"));
	    mergeType.addComponent("Separate files", DataboardForm.directoryDialog());
	    outputOptions.addComponent("Output", mergeType);
	    UnionType publisherType = UnionType.newEnum("File", "Dir", "Zip", "Smartplant Foundation", "Webserver");
	    outputOptions.addComponent("Publish to", publisherType);	    

	    // Output format
	    RecordType outputOptions2 = new RecordType(); 
	    RecordType tabType = new RecordType();	    
	    tabType.metadata.put("style", "tabbed");
	    tabType.addComponent("File", fileOptions);
	    tabType.addComponent("Dir", dirOptions);
	    tabType.addComponent("Zip", zipOptions);
	    outputOptions2.addComponent("Output2", tabType);
	    
	    // Diagram layers
	    RecordType diagramOptions = new RecordType();
	    Datatype second = new DoubleType("s");
	    
	    RecordType layers = new RecordType();
	    layers.metadata.put("style", "dialog");
	    layers.addComponent("Layer 1", Datatypes.BOOLEAN);
	    layers.addComponent("Layer 2", Datatypes.BOOLEAN);
	    layers.addComponent("Layer 3", Datatypes.BOOLEAN);
	    diagramOptions.addComponent("Layers", layers);
	    
	    // Diagram PDF Export Format options
	    diagramOptions.addComponent("Fit to content", Datatypes.BOOLEAN);

	    // Experiment options
	    RecordType experimentOptions = new RecordType();	    
	    experimentOptions.addComponent("Start Time", second);	    
	    experimentOptions.addComponent("End Time", second);	    
	    experimentOptions.addComponent("Experiment", experiments);	    
	    
	    // CSV Export
	    RecordType csvOptions = new RecordType();	    
	    csvOptions.addComponent("Step Size", second);

	    // Chart options
	    RecordType chartOptions = new RecordType();	    
	    chartOptions.addComponent("Autoscale", UnionType.newEnum("Stacked", "Overlapping"));
	    chartOptions.addComponent("TimeFormat", UnionType.newEnum("Decimal", "Time"));
	    chartOptions.addComponent("ValueFormat", UnionType.newEnum("Currency", "Scientific", "Engineering", "Default"));
	    
	    // PDF options
	    RecordType pdfOptions = new RecordType();
	    // PDF metadata	    
	    pdfOptions.addComponent("Title", Datatypes.STRING );
	    pdfOptions.addComponent("Author", Datatypes.STRING );
	    pdfOptions.addComponent("Subject", Datatypes.STRING );
	    pdfOptions.addComponent("Keywords", DataboardForm.TEXTBOX );	    
	    // PDF Encryption
	    StringType privatekeys = DataboardForm.fileOpenDialog("PKCS#12 keystore (.p12)", ".p12", "PFX (.pfx)", ".pfx"); 
	    pdfOptions.addComponent("Private Key", privatekeys);
	    pdfOptions.addComponent("Keystore Password", DataboardForm.PASSWORD);
	    pdfOptions.addComponent("Private Key Password", DataboardForm.PASSWORD);	    
	    // Formatter options 
	    pdfOptions.addComponent("Compression", UnionType.newEnum("0 (No compression)", "1", "2", "3", "4", "5", "6", "7", "8", "9 (Best)"));

	    
	    ScrolledComposite scroll = new ScrolledComposite(shell, SWT.V_SCROLL);
	    final Composite composite = new Composite(scroll, 0);
        scroll.setContent(composite);
        scroll.setExpandHorizontal(true);
        scroll.setExpandVertical(false);
	    final DataboardForm form = new DataboardForm();
	    form.setFirstColumnWidth(150);	    
	    
		composite.setLayout( new GridLayout(3, false) );			
		composite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 3, 1));			
	    
//		form.addField(composite, "File", fileOptions);
//		form.addField(composite, "Dir", dirOptions);
//		form.addField(composite, "Zip", zipOptions);
		
		form.addField(composite, "Output Options", outputOptions);
		form.addField(composite, "Output Options2", outputOptions2);
		form.addField(composite, "Diagram", diagramOptions);
		form.addField(composite, "Experiment", experimentOptions);
		form.addField(composite, "Comma Separated Value (CSV)", csvOptions);
		form.addField(composite, "Chart", chartOptions);
		form.addField(composite, "Portable Document Format (PDF)", pdfOptions);
		
		form.addListener(composite, form.type(), new Listener() {
			public void handleEvent(Event event) {
				System.out.println(event);
			} } );
		
		// Find control by reference
		ChildReference ref = ChildReference.parsePath("Output Options/Publish to");
		Control c = form.getControl(composite, ref );
		System.out.println( ref+" = "+c );
		
		// Button that validates all fields
		Button validate = new Button(composite, SWT.DEFAULT);
		validate.setText( "Validate" );
		validate.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				List<Problem> problems = form.validate(composite);
				StringBuilder sb = new StringBuilder();
				if ( problems.isEmpty() ) {
					sb.append("No problemo!");
				} else {
					for (Problem problem : problems) {
						sb.append(problem.fieldReference+": "+problem.error);
						sb.append("\n");
					}
				}
					
				MessageBox dialog = new MessageBox(shell, SWT.ICON_QUESTION | SWT.OK| SWT.CANCEL);
				dialog.setText("Form");
				dialog.setMessage( sb.toString() );
				dialog.open(); 										
			}
		});
		
		
		// Button that reads all fields
		Button read = new Button(composite, SWT.DEFAULT);
		read.setText( "Read" );
		read.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
			    // Read result values
				try {
				    RecordBinding binding = (RecordBinding) Bindings.getMutableBinding( form.type() );
				    Object formValues = binding.createDefault();
					form.readFields(composite, binding, formValues);
					String txt = binding.toString(formValues, false);
					
					MessageBox dialog = new MessageBox(shell, SWT.ICON_QUESTION | SWT.OK| SWT.CANCEL);
					dialog.setText("Form");
					dialog.setMessage(txt);
					dialog.open(); 					
					
				} catch (AccessorConstructionException e1) {
					e1.printStackTrace();
				} catch (AccessorException e1) {
					e1.printStackTrace();
				} catch (BindingException e1) {
					e1.printStackTrace();
				}
			}
		});

		// Button that writes fields
		Button write = new Button(composite, SWT.DEFAULT);
		write.setText( "Write" );
		write.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				try {
					RecordType type = form.type();
				    RecordBinding binding = (RecordBinding) Bindings.getBinding( type );				    
				    Object values = binding.createRandom( new Random() );
					form.writeFields(composite, binding, values);
				} catch (AccessorConstructionException e1) {
					e1.printStackTrace();
				} catch (AccessorException e1) {
					e1.printStackTrace();
				} catch (BindingException e1) {
					e1.printStackTrace();
				}
				
			}
		});
		
		Button modal = new Button(composite, SWT.DEFAULT);
		modal.setText("Modal");
		modal.addSelectionListener( new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				Shell dialogShell = new Shell(display, SWT.DIALOG_TRIM);
				// populate dialogShell
				dialogShell.open();
				while (!dialogShell.isDisposed()) {
				    if (!display.readAndDispatch()) {
				        display.sleep();
				    }
				}
			}
		});
					
	    composite.pack();
	    
	    shell.setSize(800,400);
	    //shell.layout();
	    shell.open ();
	    while (!shell.isDisposed()) {
	        if (!display.readAndDispatch ()) display.sleep ();
	    }
	    
	    display.dispose ();
	}
		
}
