/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.format;

import java.math.BigDecimal;
import java.math.MathContext;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.FieldPosition;
import java.text.Format;
import java.text.NumberFormat;
import java.text.ParsePosition;
import java.util.Locale;

/**
 * @author Antti Villberg
 */
public class FormattingUtils {

    public static final FormattingUtil US      = new FormattingUtil(5, 5, Locale.US);
    public static final FormattingUtil DEFAULT = US;

    public static String engineeringFormat(Object value) {
        return DEFAULT.engineeringFormat(value);
    }

    public static Format significantDigitFormat(int withSignificantDigits) {
        return significantDigitFormat(Locale.getDefault(Locale.Category.FORMAT), withSignificantDigits);
    }

    public static Format significantDigitFormat(Locale locale, int withSignificantDigits) {
        return significantDigitFormat(locale, withSignificantDigits, false);
    }

    public static Format significantDigitFormat(int withSignificantDigits, boolean stripTrailingZeros) {
        return significantDigitFormat(Locale.getDefault(Locale.Category.FORMAT), withSignificantDigits, stripTrailingZeros);
    }

    public static Format significantDigitFormat(Locale locale, int withSignificantDigits, boolean stripTrailingZeros) {
        if (withSignificantDigits < 1)
            throw new IllegalArgumentException("withSignificantDigits must be > 0, got " + withSignificantDigits);
        StringBuilder sb = new StringBuilder(withSignificantDigits + 3);
        sb.append("0.");
        for (int i = 0; i < withSignificantDigits-1; i++)
            sb.append("#");
        sb.append("E0");
        NumberFormat low = new DecimalFormat(sb.toString(), DecimalFormatSymbols.getInstance(locale));
        low.setGroupingUsed(false);
        NumberFormat hi = new SignificantDigitFormat(locale, withSignificantDigits, stripTrailingZeros);
        Format format = new SwitchFormat(0.1, 1, low, hi, hi);
        return format;
    }

    private static class SignificantDigitFormat extends NumberFormat {

        private static final long serialVersionUID = 1079268918077141671L;

        private MathContext mc;
        private boolean stripTrailingZeros;
        private boolean fixDecimalSeparator;
        private char decimalSeparator;

        public SignificantDigitFormat(Locale locale, int digits, boolean stripTrailingZeros) {
            this.mc = new MathContext(digits);
            this.stripTrailingZeros = stripTrailingZeros;
            DecimalFormatSymbols symbols = DecimalFormatSymbols.getInstance(locale);
            decimalSeparator = symbols.getDecimalSeparator();
            // BigDecimal always formats doubles with '.' as decimal separator.
            if (decimalSeparator != '.')
                fixDecimalSeparator = true;
            else
                fixDecimalSeparator = false;
        }

        @Override
        public StringBuffer format(double number, StringBuffer toAppendTo, FieldPosition pos) {
            BigDecimal big = new BigDecimal(number, mc);
            if (stripTrailingZeros)
                big = big.stripTrailingZeros();
            String str = big.toString();
            if (fixDecimalSeparator)
                str = str.replace('.', decimalSeparator);
            return toAppendTo.append(str);
        }

        @Override
        public StringBuffer format(long number, StringBuffer toAppendTo, FieldPosition pos) {
            return toAppendTo.append(number);
        }

        @Override
        public Number parse(String source, ParsePosition parsePosition) {
            return 0;
        }

    }

}
