/*******************************************************************************
 * Copyright (c) 2007- VTT Technical Research Centre of Finland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
/*
 * Created on Jan 21, 2005
 * 
 * Copyright Toni Kalajainen
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.simantics.utils.bytes;

/**
 * Big Endian long <-> byte array conversions
 * Motorola order, Network order
 *
 * @author Toni Kalajainen
 */
public class BELong {
	
	/**
	 * Convert long to byte array
	 * @param l long value
	 * @return byte array
	 */
	public static byte[] toBytes(long value)
	{
		byte array[] = new byte[8];
		array[0] = (byte) (value & 0xff);
		array[1] = (byte) ((value >> 8) & 0xff);
		array[2] = (byte) ((value >> 16) & 0xff);
		array[3] = (byte) ((value >> 24) & 0xff);
		array[4] = (byte) ((value >> 32) & 0xff);
		array[5] = (byte) ((value >> 40) & 0xff);
		array[6] = (byte) ((value >> 48) & 0xff);
		array[7] = (byte) ((value >> 56) & 0xff);
		return array;
	}
	
	/**
	 * Write long value to byte array
	 * @param value the long value
	 * @param array the byte array
	 * @param offset the offset
	 */
	public static void write(long value, byte array[], int offset)
	{
		if (offset+8>array.length)
			throw new IndexOutOfBoundsException();		
		array[0 + offset] = (byte) (value & 0xff);
		array[1 + offset] = (byte) (value >> 8);
		array[2 + offset] = (byte) (value >> 16);
		array[3 + offset] = (byte) (value >> 24);
		array[4 + offset] = (byte) (value >> 32);
		array[5 + offset] = (byte) (value >> 40);
		array[6 + offset] = (byte) (value >> 48);
		array[7 + offset] = (byte) (value >> 56);
	}
	
	/**
	 * Write long value to byte array
	 * @param value the long value
	 * @param array the byte array
	 * @param offset the offset
	 */
	public static void write(long value, byte array[])
	{
		if (array.length<8)
			throw new IndexOutOfBoundsException();		
		array[0] = (byte) (value & 0xff);
		array[1] = (byte) (value >> 8);
		array[2] = (byte) (value >> 16);
		array[3] = (byte) (value >> 24);
		array[4] = (byte) (value >> 32);
		array[5] = (byte) (value >> 40);
		array[6] = (byte) (value >> 48);
		array[7] = (byte) (value >> 56);
	}	
	/**
	 * read long value from byte array
	 * @param array the array
	 * @param offset offset
	 * @return the value
	 */
	public static long toLong(byte array[], int offset)
	{
		if (offset+8>array.length)
			throw new IndexOutOfBoundsException();		
		long value = 
			( ((long) array[0 + offset] & 0xFF) ) |
			( ((long) array[1 + offset] & 0xFF) << 8) |
			( ((long) array[2 + offset] & 0xFF) << 16) | 
			( ((long) array[3 + offset] & 0xFF) << 24) | 
			( ((long) array[4 + offset] & 0xFF) << 32) | 
			( ((long) array[5 + offset] & 0xFF) << 40) | 
			( ((long) array[6 + offset] & 0xFF) << 48) | 
			( ((long) array[7 + offset] & 0xFF) << 56);
		return value;
	}
	/**
	 * read long value from byte array
	 * @param array the array
	 * @return the value
	 */
	public static long toLong(byte array[])
	{
		if (8>array.length)
			throw new IndexOutOfBoundsException();		
		long value = 
			( ((long) array[0] & 0xFF) ) |
			( ((long) array[1] & 0xFF) << 8) |
			( ((long) array[2] & 0xFF) << 16) | 
			( ((long) array[3] & 0xFF) << 24) | 
			( ((long) array[4] & 0xFF) << 32) | 
			( ((long) array[5] & 0xFF) << 40) | 
			( ((long) array[6] & 0xFF) << 48) | 
			( ((long) array[7] & 0xFF) << 56);
		return value;
	}

	/**
	 * Test cases
	 * @param args
	 */
	public static void main(String[] args) {
		System.out.println("min="+Long.MIN_VALUE+" max="+Long.MAX_VALUE);
		long value = -1290000000;
		byte array[] = toBytes(value);
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
		write(value, array, 0);
		System.out.print(value);
		System.out.print(" = ");
		printByteArray(array);
		System.out.println();
		
		value = toLong(array, 0);
		printByteArray(array);
		System.out.print(" = ");
		System.out.print(value);
		System.out.println();
				
		value = toLong(array);
		printByteArray(array);
		System.out.print(" = ");
		System.out.print(value);
		System.out.println();
		
	}
	
	public static void printByteArray(byte array[]) {
		for (int i=0; i<array.length; i++) {
			System.out.print(array[i] & 0xff);
			if (i<array.length-1) 
				System.out.print(",");
		}
	}
}
