package org.simantics.structural.synchronization.base;

import org.eclipse.core.runtime.IProgressMonitor;
import org.simantics.db.ReadGraph;
import org.simantics.db.exception.CancelTransactionException;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.structural.synchronization.utils.ComponentBase;

import gnu.trove.map.hash.THashMap;

/**
 * Utility for updating UIDs in the given component to correspond
 * the RVIs of the given variable. This operation is needed when
 * the mapping is part of a model that is imported to a new database
 * and resource ids have been changed.
 *
 * @author Hannu Niemist&ouml;
 * @author Tuukka Lehtonen
 */
public class UpdateComponentUids<T extends ComponentBase<T>> {

    public static interface ComponentUpdateProgressMonitor {
        void componentsDone(int components, int componentCount);
    }

    private THashMap<String, String> oldToNewUids;
    private IProgressMonitor monitor;
    private ComponentUpdateProgressMonitor componentMonitor;
    private ReadGraph graph;
    private int componentCount;
    private int counter;

    private UpdateComponentUids(IProgressMonitor monitor, ReadGraph graph, int componentCount) {
        this.monitor = monitor;
        this.componentMonitor = monitor instanceof ComponentUpdateProgressMonitor ? (ComponentUpdateProgressMonitor) monitor : null;
        this.graph = graph;
        this.componentCount = componentCount;
        this.oldToNewUids = new THashMap<>(componentCount);
    }

    private void update(T component, Variable variable) throws DatabaseException {
        String newUid = variable.getRVI(graph).toString();
        oldToNewUids.put(component.uid, newUid);
        component.uid = newUid;

        // Handle progress monitoring and cancellation
        counter++;
        if (monitor != null) {
            if (componentMonitor != null)
                componentMonitor.componentsDone(counter, componentCount);
            if (monitor.isCanceled())
                throw new CancelTransactionException();
        }

        for(Variable childVariable : variable.getChildren(graph)) {
            String childName = childVariable.getName(graph);
            T childComponent = component.getChild(childName);
            if(childComponent != null)
                update(childComponent, childVariable);
        }
    }

    public static <T extends ComponentBase<T>> THashMap<String, String> update(ReadGraph graph, T component, Variable variable) throws DatabaseException {
        return update(null, graph, component, variable);
    }

    public static <T extends ComponentBase<T>> THashMap<String, String> update(IProgressMonitor monitor, ReadGraph graph, T component, Variable variable) throws DatabaseException {
        UpdateComponentUids<T> updateComponentUids = new UpdateComponentUids<T>(monitor, graph, countComponents(component));
        updateComponentUids.update(component, variable);
        return updateComponentUids.oldToNewUids;
    }

    public static <T extends ComponentBase<T>> int countComponents(T component) {
        return new Counter<T>().count(component);
    }

    private static class Counter<T extends ComponentBase<T>> {
        int counter;
        int count(T component) {
            ++counter;
            for (T child : component.getChildren())
                count(child);
            return counter;
        }
    }

}
