package org.simantics.spreadsheet.graph;

import java.util.HashSet;
import java.util.Set;

import org.simantics.spreadsheet.Range;
import org.simantics.spreadsheet.graph.parser.ast.AstApply;
import org.simantics.spreadsheet.graph.parser.ast.AstArgList;
import org.simantics.spreadsheet.graph.parser.ast.AstArithmeticExpression;
import org.simantics.spreadsheet.graph.parser.ast.AstArray;
import org.simantics.spreadsheet.graph.parser.ast.AstArrayFormulaReference;
import org.simantics.spreadsheet.graph.parser.ast.AstBoolean;
import org.simantics.spreadsheet.graph.parser.ast.AstDouble;
import org.simantics.spreadsheet.graph.parser.ast.AstFactor;
import org.simantics.spreadsheet.graph.parser.ast.AstIdentifier;
import org.simantics.spreadsheet.graph.parser.ast.AstInteger;
import org.simantics.spreadsheet.graph.parser.ast.AstNothing;
import org.simantics.spreadsheet.graph.parser.ast.AstNull;
import org.simantics.spreadsheet.graph.parser.ast.AstRange;
import org.simantics.spreadsheet.graph.parser.ast.AstRelation;
import org.simantics.spreadsheet.graph.parser.ast.AstString;
import org.simantics.spreadsheet.graph.parser.ast.AstTerm;
import org.simantics.spreadsheet.graph.parser.ast.AstValue;
import org.simantics.spreadsheet.graph.parser.ast.AstValueVisitor;
import org.simantics.spreadsheet.util.SpreadsheetUtils;

public class PrintVisitor implements AstValueVisitor<String> {

	@Override
	public String visit(AstBoolean astBoolean) {
		return "" + astBoolean.value;
	}

	@Override
	public String visit(AstDouble astFloat) {
		return "" + astFloat.value;
	}

	@Override
	public String visit(AstInteger astInteger) {
		return "" + astInteger.value;
	}

	@Override
	public String visit(AstNull astNull) {
		return "AstNull";
	}

	@Override
	public String visit(AstString astString) {
		return "\"" + astString.value + "\"";  
	}

	@Override
	public String visit(AstRange astRange) {
		if(astRange.isCell()) {
			return astRange.first;
		} else {
			return astRange.first + ":" + astRange.second;
		}
	}

	@Override
	public String visit(AstArgList astArgList) {
		StringBuilder b = new StringBuilder();
		for(int i=0;i<astArgList.values.size();i++) {
			if(i > 0) b.append(";");
			b.append(astArgList.values.get(i).accept(this));
		}
		return b.toString();
	}

	@Override
	public String visit(AstApply astApply) {
		if(astApply.args == null) {
			return astApply.value + "()";
		} else {
			return astApply.value + "(" + astApply.args.accept(this) + ")";
		}
	}

	@Override
	public String visit(AstRelation astRelation) {

		StringBuilder b = new StringBuilder();
		b.append(astRelation.left.accept(this));
		b.append(astRelation.op.trim());
		b.append(astRelation.right.accept(this));
		return b.toString();
		
	}
	
	@Override
	public String visit(AstArithmeticExpression exp) {
		
		StringBuilder b = new StringBuilder();
		if(exp.prefix != null) b.append(exp.prefix);
		
		b.append(exp.left.accept(this));

		for(int i=0;i<exp.rightCount();i++) {
			String op = exp.rightOp(i);
			b.append(op);
			AstValue value = exp.rightValue(i);
			b.append(value.accept(this));
		}
		
		return b.toString();
		
	}

	@Override
	public String visit(AstTerm exp) {
		
		StringBuilder b = new StringBuilder();
		
		b.append(exp.left.accept(this));

		for(int i=0;i<exp.rightCount();i++) {
			String op = exp.rightOp(i);
			b.append(op);
			AstValue value = exp.rightValue(i);
			b.append(value.accept(this));
		}
		
		return b.toString();
		
	}

	@Override
	public String visit(AstFactor exp) {
		
		StringBuilder b = new StringBuilder();
		
		b.append(exp.left.accept(this));

		for(int i=0;i<exp.rightCount();i++) {
			String op = exp.rightOp(i);
			b.append(op);
			AstValue value = exp.rightValue(i);
			b.append(value.accept(this));
		}
		
		return b.toString();
		
	}

	@Override
	public String visit(AstIdentifier id) {
		return "AstIdentifier";
	}
	
	@Override
	public String visit(AstArray array) {
		StringBuilder b = new StringBuilder();
		for(int i=0;i<array.values.size();i++) {
			if(i > 0) b.append(";");
			b.append(array.values.get(i).accept(this));
		}
		return b.toString();
	}
	
	@Override
	public String visit(AstNothing array) {
		return "AstNothing";
	}
	
	@Override
	public String visit(AstArrayFormulaReference ref) {
		return "{" + ref.value.accept(this) + "}";
	}

}
