package org.simantics.scl.runtime.reporting;

import java.io.FileWriter;
import java.io.IOException;

import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.function.Function;
import org.simantics.scl.runtime.tuple.Tuple0;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Static methods for calling the currently active {@link SCLReportingHandler}.
 * 
 * @author Hannu Niemist&ouml;
 */
public class SCLReporting {

    private static final Logger LOGGER = LoggerFactory.getLogger(SCLReporting.class);

    public static void print(String text) {
        SCLReportingHandler handler = ((SCLReportingHandler)SCLContext.getCurrent().get(SCLReportingHandler.REPORTING_HANDLER));
        if(handler != null)
            handler.print(text);
        else
            System.out.println(text);
    }
    
    public static void printError(String text) {
        SCLReportingHandler handler = ((SCLReportingHandler)SCLContext.getCurrent().get(SCLReportingHandler.REPORTING_HANDLER));
        if(handler != null)
            handler.printError(text);
        else
            LOGGER.error(text);
    }
    
    public static void didWork(double amount) {
        SCLReportingHandler handler = ((SCLReportingHandler)SCLContext.getCurrent().get(SCLReportingHandler.REPORTING_HANDLER));
        if(handler != null)
            handler.didWork(amount);
    }
    
    private static class FileReportingHandler extends DelegatingSCLReportingHandler {
        FileWriter writer;
        
        public FileReportingHandler(SCLReportingHandler baseHandler, FileWriter writer) {
            super(baseHandler);
            this.writer = writer;
        }

        @Override
        public void print(String text) {
            try {
                writer.append(text);
                writer.append('\n');
            } catch(IOException e) {
                e.printStackTrace();
            }
        }
    }

    public static Object printingToFile(String fileName, Function proc) throws IOException {
        FileWriter writer = new FileWriter(fileName);
        SCLContext context = SCLContext.getCurrent();
        Object oldHandler = context.get(SCLReportingHandler.REPORTING_HANDLER);
        context.put(SCLReportingHandler.REPORTING_HANDLER,
                new FileReportingHandler(
                        oldHandler == null ? SCLReportingHandler.DEFAULT : (SCLReportingHandler)oldHandler,
                        writer));
        try {
            return proc.apply(Tuple0.INSTANCE);
        } finally {
            context.put(SCLReportingHandler.REPORTING_HANDLER, oldHandler);
            writer.close();
        }
    }
    
    public static Object printErrorsAsNormalPrints(Function proc) {
        SCLContext context = SCLContext.getCurrent();
        SCLReportingHandler handler = (SCLReportingHandler)context.get(SCLReportingHandler.REPORTING_HANDLER);
        if(handler == null)
            handler = SCLReportingHandler.DEFAULT;
        
        context.put(SCLReportingHandler.REPORTING_HANDLER, new DelegatingSCLReportingHandler(handler) {
            @Override
            public void printError(String error) {
                print(error);
            }
        });
        
        try {
            return proc.apply(Tuple0.INSTANCE);
        } finally {
            context.put(SCLReportingHandler.REPORTING_HANDLER, handler);
        }
    }
    
    public static Object disablePrintingForCommand(Function proc) {
        SCLContext context = SCLContext.getCurrent();
        SCLReportingHandler handler = (SCLReportingHandler)context.get(SCLReportingHandler.REPORTING_HANDLER);
        if(handler == null)
            handler = SCLReportingHandler.DEFAULT;
        
        context.put(SCLReportingHandler.REPORTING_HANDLER, new DelegatingSCLReportingHandler(handler) {
            @Override
            public void print(String text) {
            }
        });
        
        try {
            return proc.apply(Tuple0.INSTANCE);
        } finally {
            context.put(SCLReportingHandler.REPORTING_HANDLER, handler);
        }
    }
    
    public static Object printingToLogging(Function proc) {
        SCLContext context = SCLContext.getCurrent();
        SCLReportingHandler handler = (SCLReportingHandler)context.get(SCLReportingHandler.REPORTING_HANDLER);
        if(handler == null)
            handler = SCLReportingHandler.DEFAULT;
        
        context.put(SCLReportingHandler.REPORTING_HANDLER, SCLReportingHandler.DEFAULT);
        
        try {
            return proc.apply(Tuple0.INSTANCE);
        } finally {
            context.put(SCLReportingHandler.REPORTING_HANDLER, handler);
        }
    }
}
