package org.simantics.scl.runtime.chr;

/**
 * This class implements a pairing heap of CHR priorities.
 */
public abstract class CHRPriority implements Comparable<CHRPriority> {
    
    public final int priority;
    
    // Pairing heap
    private CHRPriority sibling;
    private CHRPriority child;
    
    boolean inContext;
    
    public CHRPriority(int priority) {
        this.priority = priority;
    }
    
    /**
     * This method assume that a and b are roots and their sibling field
     * can be overridden. 
     */
    public static CHRPriority merge(CHRPriority a, CHRPriority b) {
        if(a.priority <= b.priority) {
            a.sibling = null;
            b.sibling = a.child;
            a.child = b;
            return a;
        }
        else {
            a.sibling = b.child;
            b.sibling = null;
            b.child = a;
            return b;
        }
    }
    
    protected void ensureInContext(CHRContext context) {
        if(!inContext) {
            CHRPriority topPriority = context.topPriority;
            if(topPriority == null)
                context.topPriority = this;
            else
                context.topPriority = merge(topPriority, this);
            inContext = true;
        }
    }
    
    private CHRPriority mergeSiblings() {
        if(sibling == null)
            return this;
        CHRPriority nextSibling = sibling.sibling;
        CHRPriority merged = merge(this, sibling);
        if(nextSibling == null)
            return merged;
        else
            return merge(merged, nextSibling.mergeSiblings());
    }
    
    public CHRPriority nextPriority() {
        if(child == null)
            return null;
        else {
            CHRPriority result = child.mergeSiblings();
            child = null;
            return result;
        }
    }
    
    @Override
    public int compareTo(CHRPriority o) {
        return Double.compare(priority, o.priority);
    }
    
    public abstract void activate(CHRContext context);
    //public abstract CHRRuntimeRuleset getParent();
}
