package org.simantics.scl.db;

import java.io.IOException;

import org.cojen.classfile.TypeDesc;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.VirtualGraph;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.procedure.adapter.SyncListenerAdapter;
import org.simantics.db.common.procedure.adapter.TransientCacheAsyncListener;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.common.request.DelayedWriteRequest;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.common.request.WriteResultRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.db.layer0.variable.Variables;
import org.simantics.db.request.Read;
import org.simantics.db.service.ClusterControl;
import org.simantics.db.service.QueryControl;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.db.service.VirtualGraphSupport;
import org.simantics.layer0.utils.triggers.IActivationManager;
import org.simantics.scl.compiler.environment.specification.EnvironmentSpecification;
import org.simantics.scl.compiler.errors.Failable;
import org.simantics.scl.compiler.internal.codegen.types.JavaTypeTranslator;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.repository.ImportFailureException;
import org.simantics.scl.compiler.runtime.RuntimeEnvironment;
import org.simantics.scl.compiler.types.Type;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.runtime.function.Function;
import org.simantics.scl.runtime.function.Function1;
import org.simantics.scl.runtime.reporting.SCLReportingHandler;
import org.simantics.scl.runtime.tuple.Tuple;
import org.simantics.scl.runtime.tuple.Tuple0;
import org.simantics.utils.DataContainer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@SuppressWarnings({"rawtypes", "unchecked"})
public class SCLFunctions {

    private static final Logger LOGGER = LoggerFactory.getLogger(SCLFunctions.class);

    public static final String GRAPH = "graph";

    public static <T> T safeExec(final Function f) {
        try {
            return (T)f.apply(Tuple0.INSTANCE);
        } catch (Throwable t) {
            LOGGER.error("safeExec caught exception", t);
            return null;
        }
    }
    
    public static void asyncRead(final Function f) throws DatabaseException {
        final SCLContext context = SCLContext.createDerivedContext();
        Simantics.getSession().asyncRequest(new ReadRequest() {
            @Override
            public void run(ReadGraph graph) throws DatabaseException {
                SCLContext.push(context);
                context.put(GRAPH, graph);
                try {
                    f.apply(Tuple0.INSTANCE);
                } finally {
                    SCLContext.pop();
                }
            }
        });
    }
    
    public static <T> T syncRead(final Function f) throws DatabaseException {
        final SCLContext context = SCLContext.getCurrent();
        Object graph = context.get(GRAPH);
        if (graph != null) {
            return (T)f.apply(Tuple0.INSTANCE);
        } else {
            return Simantics.getSession().syncRequest(new Read<T>() {
                @Override
                public T perform(ReadGraph graph) throws DatabaseException {
                    SCLContext.push(context);
                    ReadGraph oldGraph = (ReadGraph)context.put(GRAPH, graph);
                    try {
                        return (T)f.apply(Tuple0.INSTANCE);
                    } finally {
                        context.put(GRAPH, oldGraph);
                        SCLContext.pop();
                    }
                }
            });
        }
    }
    
    public static void asyncWrite(final Function f) throws DatabaseException {
        SCLContext context = SCLContext.createDerivedContext();
        if (Simantics.peekSession() != null) {
            Simantics.getSession().asyncRequest(new WriteRequest() {
                @Override
                public void perform(WriteGraph graph) throws DatabaseException {
                    SCLContext.push(context);
                    context.put(GRAPH, graph);
                    try {
                        f.apply(Tuple0.INSTANCE);
                    } finally {
                        SCLContext.pop();
                    }
                }
            });
        } else {
            LOGGER.warn("No session available for asynchronous write requests");
        }
    }
    
    public static <T> T syncWrite(final Function f) throws DatabaseException {
        final SCLContext context = SCLContext.getCurrent();
        Object graph = context.get(GRAPH);
        if (graph != null) {
            return (T)f.apply(Tuple0.INSTANCE);
        } else {
            final SCLReportingHandler printer = (SCLReportingHandler)SCLContext.getCurrent().get(SCLReportingHandler.REPORTING_HANDLER);
            return Simantics.getSession().syncRequest(new WriteResultRequest<T>() {
                @Override
                public T perform(WriteGraph graph) throws DatabaseException {
                    SCLContext.push(context);
                    SCLReportingHandler oldPrinter = (SCLReportingHandler)context.put(SCLReportingHandler.REPORTING_HANDLER, printer);
                    ReadGraph oldGraph = (ReadGraph)context.put(GRAPH, graph);
                    try {
                        return (T)f.apply(Tuple0.INSTANCE);
                    } finally {
                        context.put(GRAPH, oldGraph);
                        context.put(SCLReportingHandler.REPORTING_HANDLER, oldPrinter);
                        SCLContext.pop();
                    }
                }
            });
        }
    }
    
    public static <T> T delayedSyncWrite(final Function f) throws DatabaseException {
        final SCLContext context = SCLContext.getCurrent();
    	final DataContainer<T> dc = new DataContainer<T>(null);

        DelayedWriteRequest request = new DelayedWriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
            	final SCLContext context = SCLContext.getCurrent();
            	SCLContext.push(context);
                ReadGraph oldGraph = (ReadGraph)context.put(GRAPH, graph);
                try {
                    dc.set((T)f.apply(Tuple0.INSTANCE));
                } finally {
                    context.put(GRAPH, oldGraph);
                    SCLContext.pop();
                }
            }
        };
    	
        Object graph = context.get(GRAPH);
        if (graph != null) {
            if (graph instanceof WriteGraph) {
            	((WriteGraph)graph).syncRequest(request);
            } else {
            	throw new DatabaseException("Caller is inside a read transaction.");
            }
        } else {
            Simantics.getSession().syncRequest(request);
        }
    	return dc.get();
    }

    public static <T> T virtualSyncWriteMem(WriteGraph graph, String virtualGraphId, final Function f) throws DatabaseException {
        final SCLContext context = SCLContext.getCurrent();
        VirtualGraphSupport vgs = graph.getService(VirtualGraphSupport.class);
        VirtualGraph vg = vgs.getMemoryPersistent(virtualGraphId);
        return graph.syncRequest(new WriteResultRequest<T>(vg) {
            @Override
            public T perform(WriteGraph graph) throws DatabaseException {
                SCLContext.push(context);
                ReadGraph oldGraph = (ReadGraph)context.put(GRAPH, graph);
                try {
                    return (T)f.apply(Tuple0.INSTANCE);
                } finally {
                    context.put(GRAPH, oldGraph);
                    SCLContext.pop();
                }
            }
        });
    }
    
    public static <T> T virtualSyncWriteWS(WriteGraph graph, String virtualGraphId, final Function f) throws DatabaseException {
        final SCLContext context = SCLContext.getCurrent();
        VirtualGraphSupport vgs = graph.getService(VirtualGraphSupport.class);
        VirtualGraph vg = vgs.getWorkspacePersistent(virtualGraphId);
        return graph.syncRequest(new WriteResultRequest<T>(vg) {
            @Override
            public T perform(WriteGraph graph) throws DatabaseException {
                SCLContext.push(context);
                ReadGraph oldGraph = (ReadGraph)context.put(GRAPH, graph);
                try {
                    return (T)f.apply(Tuple0.INSTANCE);
                } finally {
                    context.put(GRAPH, oldGraph);
                    SCLContext.pop();
                }
            }
        });
    }
    
    public static <T> T readValue(final String uri) throws DatabaseException {
        return Simantics.getSession().syncRequest(new Read<T>() {
            @Override
            public T perform(ReadGraph graph) throws DatabaseException {
                return Variables.getVariable(graph, uri).getValue(graph);
            }            
        });
    }
    
    public static <T> void writeValue(final String uri, final T value) throws DatabaseException {
        Simantics.getSession().syncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                Variables.getVariable(graph, uri).setValue(graph, value);
            }            
        });
    }
    
    public static void activateOnce(Resource r) {
        Simantics.getSession().getService(IActivationManager.class).activateOnce(r);
    }
    
    public static void syncActivateOnce(WriteGraph graph, Resource r) throws DatabaseException {
        graph.getService(IActivationManager.class).activateOnce(graph, r);
    }

    public static Resource resourceFromId(ReadGraph graph, long id) throws DatabaseException, IOException {
        SerialisationSupport ss = graph.getService(SerialisationSupport.class);
        return ss.getResource(id);
    }
    
    public static void disableDependencies(WriteGraph graph) {
        Layer0Utils.setDependenciesIndexingDisabled(graph, true);       
    }
    
    public static void enableDependencies(WriteGraph graph) {
        Layer0Utils.setDependenciesIndexingDisabled(graph, false);       
    }
    
    public static void collectClusters() {
        Simantics.getSession().getService(ClusterControl.class).collectClusters(Integer.MAX_VALUE);
    }
    
    public static class SCLUnaryRead extends BinaryRead<Function1<Object,Object>, Object, Object> {

		public SCLUnaryRead(Function1<Object, Object> parameter1, Object parameter2) {
			super(parameter1, parameter2);
		}

		@Override
		public Object perform(ReadGraph graph) throws DatabaseException {
			return Simantics.applySCLRead(graph, parameter, parameter2);
		}
    	
    }
    
    public static Object unaryQuery(ReadGraph graph, Function1<Object,Object> fn, Object value) throws DatabaseException {
    	return graph.syncRequest(new SCLUnaryRead(fn, value));
    }

    public static Object unaryQueryCached(ReadGraph graph, Function1<Object,Object> fn, Object value) throws DatabaseException {
    	return graph.syncRequest(new SCLUnaryRead(fn, value), TransientCacheAsyncListener.<Object>instance());
    }
    

    private static class Subquery extends UnaryRead<Function, Object> {

        public Subquery(Function q) {
            super(q);
        }

        @Override
        public Object perform(ReadGraph graph) throws DatabaseException {
            return Simantics.applySCLRead(graph, parameter, Tuple0.INSTANCE);
        }

    }

    public static Object subquery(ReadGraph graph, Function q) throws DatabaseException {
        return graph.syncRequest(new Subquery(q));
    }

    public static Object subqueryC(ReadGraph graph, Function q) throws DatabaseException {
        return graph.syncRequest(new Subquery(q), TransientCacheAsyncListener.<Object>instance());
    }
    
    public static void subqueryL(ReadGraph graph, Function query, Function executeCallback, Function1<Throwable, Tuple> exceptionCallback, Function1<Tuple0, Boolean> isDisposedCallback) throws DatabaseException {
        graph.syncRequest(new Subquery(query), new SyncListenerAdapter<Object>() {
            @Override
            public void execute(ReadGraph graph, Object result) throws DatabaseException {
                Simantics.applySCLRead(graph, executeCallback, result);
            }
            
            @Override
            public void exception(ReadGraph graph, Throwable t) throws DatabaseException {
                Simantics.applySCLRead(graph, exceptionCallback, t);
            }
            
            @Override
            public boolean isDisposed() {
                return isDisposedCallback.apply(Tuple0.INSTANCE);
            }
        });
    }

    public static Object possibleFromDynamic(Type expectedType, String moduleName, Object value) {
    	
        try {

        	
            Failable<Module> failable = SCLOsgi.MODULE_REPOSITORY.getModule(moduleName);
            Module module = failable.getResult();
            
        	RuntimeEnvironment env = SCLOsgi.MODULE_REPOSITORY.createRuntimeEnvironment(
        			EnvironmentSpecification.of(moduleName, ""), module.getParentClassLoader());

            JavaTypeTranslator tr = new JavaTypeTranslator(env.getEnvironment());
            TypeDesc desc = tr.toTypeDesc(expectedType);
            String className = desc.getFullName();
            Class<?> clazz = env.getMutableClassLoader().loadClass(className);
            if (!clazz.isAssignableFrom(value.getClass()))
                return null;
            
        } catch (ImportFailureException | ClassNotFoundException e) {
        }
        return value;
    }

    public static void restrictQueries(ReadGraph graph, int amount, int step, int maxTimeInMs) {

		QueryControl qc = graph.getService(QueryControl.class);
		long start = System.currentTimeMillis();
		while(true) {
			int current = qc.count();
			if(current < amount) return;
			qc.gc(graph, step);
			long duration = System.currentTimeMillis() - start;
			if(duration > maxTimeInMs) return;
		}

    }

    public static int countQueries(ReadGraph graph) {

		QueryControl qc = graph.getService(QueryControl.class);
		return qc.count();

    }
    
}
