/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagramEditor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.variable.Variable;
import org.simantics.diagram.content.ConnectionUtil;
import org.simantics.diagram.flag.FlagUtil;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.issues.ontology.IssueResource;
import org.simantics.layer0.Layer0;
import org.simantics.modeling.ModelingResources;
import org.simantics.modeling.ui.Activator;
import org.simantics.structural.stubs.StructuralResource2;
import org.simantics.ui.selection.AnyResource;
import org.simantics.ui.selection.AnyVariable;
import org.simantics.ui.selection.WorkbenchSelectionContentType;
import org.simantics.ui.selection.WorkbenchSelectionElement;
import org.simantics.ui.workbench.editor.AbstractResourceEditorAdapter;
import org.simantics.utils.datastructures.Pair;
import org.simantics.utils.ui.ISelectionUtils;

/**
 * @author Tuukka Lehtonen
 */
public class OpenDiagramFromIssue extends AbstractResourceEditorAdapter {

    private static final String EDITOR_ID = "org.simantics.modeling.ui.plainDiagramEditor";

    public OpenDiagramFromIssue() {
        super("Open Diagram Containing Referenced Component", Activator.COMPOSITE_ICON);
    }

    protected String getEditorId(ReadGraph g, Resource diagram) throws DatabaseException {
        ModelingResources MOD = ModelingResources.getInstance(g);
        String preferredEditorId = g.getPossibleRelatedValue(diagram, MOD.PreferredDiagramEditorID);
        if(preferredEditorId != null)
            return preferredEditorId;
        else
            return EDITOR_ID;
    }

    @Override
    public boolean canHandle(ReadGraph g, Object input) throws DatabaseException {
        return extractContext(g, input) != null;
    }

    @Override
    public void openEditor(Object input) throws Exception {
        Simantics.getSession().asyncRequest(new ReadRequest() {
            @Override
            public void run(ReadGraph g) throws DatabaseException {
                Pair<Resource, Collection<Object>> data = extractContext(g, input);
                if (data != null)
                    OpenDiagramFromConfigurationAdapter.openEditor(g, data.first, getEditorId(g, data.first), data.second);
            }
        });
    }

    private static Pair<Resource, Collection<Object>> extractContext(ReadGraph graph, Object input) throws DatabaseException {
        Pair<Variable, Resource> p = extractInput(graph, input);
        Pair<Resource, Collection<Object>> result = p.first != null ? findConfigurationAndObjects(graph, p.first) : null;
        if (result == null && p.second != null)
            result = findConfigurationAndObjects(graph, p.second);
        return result;
    }

    protected static <T> T extractContent(ReadGraph graph, Object input, WorkbenchSelectionContentType<T> contentType, Class<T> contentClass) throws DatabaseException {
        if (contentClass.isInstance(input))
            return contentClass.cast(input);
        if (input instanceof WorkbenchSelectionElement) {
            WorkbenchSelectionElement single = (WorkbenchSelectionElement) input;
            if (single != null)
                return single.getContent(contentType);
        }
        return ISelectionUtils.filterSingleSelection(input, contentClass);
    }

    protected static Pair<Variable, Resource> extractInput(ReadGraph graph, Object input) throws DatabaseException {
        return Pair.make(
                extractContent(graph, input, new AnyVariable(graph), Variable.class),
                extractContent(graph, input, new AnyResource(graph), Resource.class));
    }

    protected static Pair<Resource, Collection<Object>> findConfigurationAndObjects(ReadGraph graph, Resource issue) throws DatabaseException {
        IssueResource ISSUE = IssueResource.getInstance(graph);
        if (!graph.isInstanceOf(issue, ISSUE.Issue))
            return null;
        List<Resource> contexts = graph.getRelatedValue2(issue, ISSUE.Issue_contexts);
        return contexts != null ? findConfigurationAndObjects(graph, contexts) : null;
    }

    protected static Pair<Resource, Collection<Object>> findConfigurationAndObjects(ReadGraph graph, Variable v) throws DatabaseException {
        List<Resource> contexts = v.getPossiblePropertyValue(graph, IssueResource.getInstance(graph).Issue_contexts);
        return contexts != null ? findConfigurationAndObjects(graph, contexts) : null;
    }

    protected static Pair<Resource, Collection<Object>> findConfigurationAndObjects(ReadGraph graph, List<Resource> contexts) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        DiagramResource DIA = DiagramResource.getInstance(graph);
        StructuralResource2 STR = StructuralResource2.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);

        for(Resource context : contexts) {
            Set<Resource> types = graph.getTypes(context);
            if (types.contains(DIA.Element)) {
                Resource config = findConfigurationForElement(graph, context);
                if (config != null) {
                    Collection<Object> elements = Collections.<Object>singleton(context);
                    return Pair.make(config, elements);
                }
            } else if (types.contains(STR.Component) && !types.contains(STR.Composite)) {
                Resource config = graph.getPossibleObject(context, L0.PartOf);
                if (config != null) {
                    return Pair.make(config, findElementObjects(graph, context));
                }
            } else if (types.contains(STR.Connection)) {
                Resource element = graph.getPossibleObject(context, MOD.ConnectionToDiagramConnection);
                if (element != null) {
                    Resource config = findConfigurationForElement(graph, element);
                    if (config != null) {
                        return Pair.make(config, Collections.<Object>singleton(element));
                    }
                }
            }
        }

        return null;
    }

    protected static Resource findConfigurationForElement(ReadGraph graph, Resource element) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        ModelingResources MOD = ModelingResources.getInstance(graph);
        Resource diagram = graph.getPossibleObject(element, L0.PartOf);
        if (diagram == null) return null;
        return graph.getPossibleObject(diagram, MOD.DiagramToComposite);
    }

    protected static Collection<Object> findElementObjects(ReadGraph g, Resource component) throws DatabaseException {
        Collection<Object> result = findElementObjects(g, component, "");
        ModelingResources MOD = ModelingResources.getInstance(g);
        for (Resource element : g.getObjects(component, MOD.HasParentComponent_Inverse))
            result.add(element);
        return result;
    }
    
    public static Collection<Object> findElementObjects(ReadGraph g, Resource component, String rviFromComponent) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(g);
        DiagramResource DIA = DiagramResource.getInstance(g);
        ModelingResources MOD = ModelingResources.getInstance(g);
        final Collection<Object> selectedObjects = new ArrayList<>(4);
        if (rviFromComponent.isEmpty()) {
            // The selected objects are configuration objects
            for (Resource element : g.getObjects(component, MOD.ComponentToElement)) {
                if (g.isInstanceOf(element, DIA.Flag) && FlagUtil.isExternal(g, element)) {
                    // Use external flag primarily if one exists in the correspondences
                    selectedObjects.clear();
                    selectedObjects.add(element);
                    break;
                } else if (g.isInstanceOf(element, DIA.RouteGraphConnection)) {
                    selectedObjects.add(element);
                } else if (g.isInstanceOf(element, DIA.Connection)) {
                    // Okay, we need to find a part of the connection
                    ConnectionUtil cu = new ConnectionUtil(g);
                    cu.gatherConnectionParts(element, selectedObjects);
                } else {
                    selectedObjects.add(element);
                }
            }
        } else {
            // The selected objects are generated components
            for (Resource refElement : g.getObjects(component, MOD.HasParentComponent_Inverse)) {
                Resource relation = g.getPossibleObject(refElement, MOD.HasReferenceRelation);
                if (relation != null) {
                    String suffix = g.getPossibleRelatedValue(relation, L0.HasName, Bindings.STRING);
                    if (suffix != null) {
                        if (rviFromComponent.equals(suffix)) {
                            selectedObjects.add(refElement);
                        }
                    }
                }
            }
        }
        return selectedObjects;
    }
    
}
