/*******************************************************************************
 * Copyright (c) 2012 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.diagramEditor;

import java.util.Collections;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubMonitor;
import org.simantics.DatabaseJob;
import org.simantics.db.exception.CancelTransactionException;
import org.simantics.g2d.diagram.DiagramHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.modeling.ui.Activator;
import org.simantics.utils.DataContainer;
import org.simantics.utils.threads.ThreadUtils;

public class DiagramViewerLoadJob extends DatabaseJob {

    private static final boolean PROFILE = false;
    private DiagramViewer        viewer;

    public DiagramViewerLoadJob(DiagramViewer viewer) {
        super("Load Diagram");
        setUser(true);
        this.viewer = viewer;
    }

    @Override
    protected IStatus run(final IProgressMonitor monitor) {
        final SubMonitor mon = SubMonitor.convert(monitor, "Loading Diagram", 200);

        try {
            Object task = BEGIN("DV.loadDiagram");
            final IDiagram diagram = viewer.loadDiagram(mon.newChild(100), viewer.diagramResource);
            if (diagram == null)
                return Status.CANCEL_STATUS;
            END(task);

            // Start an activation for the input resource.
            // This will activate mapping if necessary.
            task = BEGIN("DV.performActivation");
            viewer.performActivation(mon.newChild(50));
            END(task);

            // Wait for load completion in AWT thread
            ThreadUtils.syncExec(viewer.canvasContext.getThreadAccess(), new Runnable() {
                @Override
                public void run() {
                    setThread(viewer.canvasContext.getThreadAccess().getThread());
                    mon.setTaskName("Finalize Diagram Loading");

                    try {
                        Object task = BEGIN("DV.beforeSetDiagram");
                        viewer.beforeSetDiagram(diagram);
                        mon.worked(10);
                        END(task);

                        task = BEGIN("DV.setDiagramHint");
                        mon.subTask("Set Diagram");
                        DataContainer<IDiagram> diagramContainer = viewer.sourceDiagramContainer;
                        if (diagramContainer != null)
                            diagramContainer.set( diagram );
                        viewer.sourceDiagram = diagram;
                        viewer.canvasContext.getDefaultHintContext().setHint(DiagramHints.KEY_DIAGRAM, diagram);
                        mon.worked(10);
                        END(task);

                    	viewer.selectionProvider.fireSelection(Collections.emptyList());
                        
                        // Zoom to fit if no previous view transform is available
                        task = BEGIN("DV.scheduleZoomToFit");
                        viewer.scheduleZoomToFit(diagram);
                        mon.worked(10);
                        END(task);

                        task = BEGIN("DV.onCreated");
                        mon.subTask("");
                        viewer.onCreated();
                        mon.worked(10);
                        END(task);

                        task = BEGIN("DV.applyEditorState");
                        mon.subTask("Apply editor state");
                        viewer.applyEditorState(viewer.editorState, viewer.canvasContext);
                        mon.worked(10);
                        END(task);

                        task = BEGIN("DV.activateUiContexts");
                        viewer.contextUtil.inContextThread(new Runnable() {
                            @Override
                            public void run() {
                                if (!viewer.disposed)
                                    viewer.activateUiContexts(viewer.contextUtil);
                                viewer = null;
                            }
                        });
                        END(task);
                    } catch (Throwable t) {
                        viewer = null;
                    }
                }
            });

            return Status.OK_STATUS;
        } catch (CancelTransactionException e) {
            monitor.done();
            viewer = null;
            return new Status(IStatus.CANCEL, Activator.PLUGIN_ID, "Diagram loading was cancelled.", e);
        } catch (Throwable t) {
            monitor.done();
            viewer = null;
            return new Status(IStatus.ERROR, Activator.PLUGIN_ID, "Diagram loading failed, see exception for details.", t);
        }
    }

    protected static Object BEGIN(String name) {
        if (PROFILE) {
            //return ThreadLog.BEGIN(name);
        }
        return null;
    }

    protected static void END(Object task) {
        if (PROFILE) {
            //((Task) task).end();
        }
    }
}
