package org.simantics.modeling.ui.diagram.renaming;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.layout.GridLayoutFactory;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ColumnLabelProvider;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ICheckStateProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.simantics.utils.ui.ISelectionUtils;

public class ComponentsRenamingDialog extends Dialog {

    ComponentsRenamingModel model;

    // UI
    Composite area;
    Label oldNamePrefixLabel;
    Text oldNamePrefix;
    Text newNamePrefix;
    Label errorLabel;
    Text error;
    CheckboxTableViewer tableViewer;

    public ComponentsRenamingDialog(Shell parentShell, ComponentsRenamingModel model) {
        super(parentShell);
        this.model = model;
        setShellStyle(SWT.RESIZE | SWT.TITLE);
    }
    
    @Override
    protected Control createDialogArea(Composite parent) {
        getShell().setText("Rename diagram contents");
        getShell().setLayout(new GridLayout());
        Composite composite = new Composite(parent, SWT.NONE);
        composite.setLayoutData(new GridData(GridData.FILL_BOTH));
        GridLayoutFactory.fillDefaults().numColumns(1).applyTo(composite);
        
        // Name prefix
        
        {
            area = new Composite(composite, SWT.BORDER);
            GridDataFactory.fillDefaults().grab(true, false).applyTo(area);
            GridLayoutFactory.fillDefaults().numColumns(2).margins(5, 5).applyTo(area);
            
            oldNamePrefixLabel = new Label(area, SWT.NONE);
            oldNamePrefixLabel.setText("Old name prefix:");
            
            oldNamePrefix = new Text(area, SWT.READ_ONLY | SWT.BORDER);
            oldNamePrefix.setEditable(false);
            oldNamePrefix.setText(model.oldNamePrefix);
            GridDataFactory.fillDefaults().grab(true, false).applyTo(oldNamePrefix);
            
            Label newNamePrefixLabel = new Label(area, SWT.NONE);
            newNamePrefixLabel.setText("&New name prefix:");
            
            newNamePrefix = new Text(area, SWT.BORDER);
            newNamePrefix.setText(model.oldNamePrefix);
            newNamePrefix.setSelection(model.oldNamePrefix.length());
            GridDataFactory.fillDefaults().grab(true, false).applyTo(newNamePrefix);
            
            newNamePrefix.addModifyListener(new ModifyListener() {

                @Override
                public void modifyText(ModifyEvent e) {
                    String text = newNamePrefix.getText();
                    if (model.prefixValidator != null) {
                        String err = model.prefixValidator.apply(text);
                        if (err != null) {
                            if (error == null) {
                                errorLabel = new Label(area, 0);
                                errorLabel.moveBelow(newNamePrefix);
                                GridDataFactory.fillDefaults().grab(false, false).applyTo(errorLabel);
                                error = new Text(area, SWT.MULTI | SWT.WRAP | SWT.READ_ONLY);
                                error.moveBelow(errorLabel);
                                GridDataFactory.fillDefaults().grab(true, false).applyTo(error);
                                error.setForeground(error.getDisplay().getSystemColor(SWT.COLOR_RED));
                            }
                            error.setText(err);
                            area.getParent().layout();
                            getButton(OK).setEnabled(false);
                            return;
                        }
                        if (error != null) {
                            errorLabel.dispose();
                            errorLabel = null;
                            error.dispose();
                            error = null;
                            area.getParent().layout();
                            getButton(OK).setEnabled(true);
                        }
                    }

                    model.newNamePrefix = text;
                    model.computeNewNames();
                    refresh();
                }
            });
        }
        
        // Reset
        
        final Button resetNames = new Button(composite, SWT.CHECK);
        resetNames.setText("&Reset names");
        resetNames.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                model.reset = resetNames.getSelection();
                model.computeNewNames();
                refresh();
            }
        });
        
        // Name table
        
        tableViewer = CheckboxTableViewer.newCheckList(composite,
                SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);

        tableViewer.getTable().setHeaderVisible(true);
        GridDataFactory.fillDefaults().grab(true, true).applyTo(tableViewer.getTable());
        
        tableViewer.setContentProvider(ArrayContentProvider.getInstance());
        tableViewer.setCheckStateProvider(new ICheckStateProvider() {
            @Override
            public boolean isGrayed(Object element) {
                return false;
            }
            @Override
            public boolean isChecked(Object element) {
                return model.selectedEntries.contains(element);
            }
        });
        
        TableViewerColumn oldNameColumn = new TableViewerColumn(tableViewer, SWT.NONE);
        oldNameColumn.getColumn().setText("Old name");
        oldNameColumn.getColumn().setWidth(250);
        oldNameColumn.setLabelProvider(new ColumnLabelProvider() {
            @Override
            public String getText(Object element) {
                NameEntry entry = (NameEntry)element;
                return entry.oldName;
            }
        });
        
        TableViewerColumn newNameColumn = new TableViewerColumn(tableViewer, SWT.NONE);
        newNameColumn.getColumn().setText("New name");
        newNameColumn.getColumn().setWidth(250);
        newNameColumn.setLabelProvider(new ColumnLabelProvider() {
            @Override
            public String getText(Object element) {
                NameEntry entry = (NameEntry)element;
                return entry.newName;
            }
        });
        tableViewer.addCheckStateListener(new ICheckStateListener() {
            void addOrRemoveSelection(NameEntry entry, boolean add) {
                if (add)
                    model.selectedEntries.add(entry);
                else
                    model.selectedEntries.remove(entry);
            }
            @Override
            public void checkStateChanged(CheckStateChangedEvent event) {
                final boolean checked = event.getChecked();
                NameEntry checkedNode = (NameEntry) event.getElement();

                Set<NameEntry> entries = new HashSet<NameEntry>();
                Set<NameEntry> selection = ISelectionUtils.filterSetSelection(tableViewer.getSelection(), NameEntry.class);
                if (selection.contains(checkedNode))
                    entries.addAll(selection);
                else
                    tableViewer.setSelection(StructuredSelection.EMPTY);
                entries.add(checkedNode);

                for (NameEntry entry : entries) {
                    addOrRemoveSelection(entry, checked);
                }

                tableViewer.refresh();
            }
        });
        tableViewer.setInput(model.entries.toArray());
        newNamePrefix.setFocus();

        // Select All/None button bar
        Composite bar = new Composite(composite, SWT.NONE);
        GridDataFactory.fillDefaults().grab(true, false).span(3, 1).applyTo(bar);
        bar.setLayout(new RowLayout());
        Button selectAll = new Button(bar, SWT.PUSH);
        selectAll.setText("Select &All");
        selectAll.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                model.selectedEntries.addAll(model.entries);
                tableViewer.setAllChecked(true);
            }
        });
        Button clearSelection = new Button(bar, SWT.PUSH);
        clearSelection.setText("&Clear Selection");
        clearSelection.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                model.selectedEntries.clear();
                tableViewer.setAllChecked(false);
            }
        });

        return composite;
    }
    
    public void refresh() {
        tableViewer.refresh();
    }
    
    public static void main(String[] args) {
        final Display display = new Display();
        Shell shell = new Shell(display);
        shell.open();
        
        ComponentsRenamingModel model = new ComponentsRenamingModel();
        model.oldNamePrefix = "FOO";
        model.newNamePrefix = "FOO";
        for(int i=0;i<100;++i)
            model.entries.add(new NameEntry(null, "FOO"+(i*5), "FOO"+(i*5), "PREFIX"));
        
        ComponentsRenamingDialog dialog = new ComponentsRenamingDialog(shell, model);
        dialog.open();
        
        while (!shell.isDisposed()) {
            if (!display.readAndDispatch())
                display.sleep();
        }
        display.dispose();
    }

}
