package org.simantics.modeling.ui.componentTypeEditor;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.AnnotationModel;
import org.eclipse.jface.text.source.IAnnotationModel;
import org.eclipse.ui.texteditor.AbstractDocumentProvider;
import org.simantics.Simantics;
import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.UniqueRead;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.util.Layer0Utils;
import org.simantics.layer0.Layer0;
import org.simantics.scl.compiler.errors.CompilationError;
import org.simantics.scl.compiler.errors.Failable;
import org.simantics.scl.compiler.errors.Failure;
import org.simantics.scl.compiler.errors.Locations;
import org.simantics.scl.compiler.module.Module;
import org.simantics.scl.compiler.module.repository.UpdateListener;
import org.simantics.scl.osgi.SCLOsgi;
import org.simantics.scl.runtime.SCLContext;
import org.simantics.scl.ui.editor.SCLSourceViewerConfigurationNew;
import org.simantics.structural2.utils.StructuralUtils;
import org.simantics.ui.workbench.ResourceEditorInput;
import org.simantics.utils.logging.TimeLogger;

public class SCLModuleEditorDocumentProvider extends AbstractDocumentProvider {

    protected Resource resource;
    protected String currentText;
    protected boolean immutable;
    protected boolean errorHappened;

    protected AnnotationModel annotationModel = new AnnotationModel();
    private final SCLSourceViewerConfigurationNew sourceViewer;

    public SCLModuleEditorDocumentProvider(SCLSourceViewerConfigurationNew sourceViewer) {
        this.sourceViewer = sourceViewer;
    }

    @Override
    protected IDocument createDocument(Object element) throws CoreException {
        ResourceEditorInput input = (ResourceEditorInput)element;
        resource = input.getResource();
        try {
            return Simantics.getSession().syncRequest(new UniqueRead<Document>() {
                @Override
                public Document perform(ReadGraph graph) throws DatabaseException {
                    Layer0 L0 = Layer0.getInstance(graph);
                    currentText = graph.getRelatedValue(resource, L0.SCLModule_definition, Bindings.STRING);
                    immutable = StructuralUtils.isImmutable(graph, resource);
                    errorHappened = false;
                    return new Document(currentText != null ? currentText : "");
                }
            });
        } catch (DatabaseException e) {
            StringWriter sw = new StringWriter();
            PrintWriter pw = new PrintWriter(sw);
            e.printStackTrace(pw);
            errorHappened = true;
            return new Document(sw.toString());
        }
    }

    // While this editor is active we do not want that this listener gets collected
    private UpdateListener listener = new UpdateListener() {
        @Override
        public void notifyAboutUpdate() {
            updateAnnotations();
        }
    };

    protected void updateAnnotations() {
        Simantics.getSession().asyncRequest(new ReadRequest() {
            @Override
            public void run(ReadGraph graph) throws DatabaseException {
                if (!graph.hasStatement(resource))
                    return;
                String moduleName = graph.getURI(resource);
                SCLContext context = SCLContext.getCurrent();
                context.put("graph", graph);
                Failable<Module> result = SCLOsgi.MODULE_REPOSITORY.getModule(moduleName, listener);
                
                if(result instanceof Failure) {
                    Failure failure = (Failure)result;
                    setAnnotations(Arrays.asList(failure.errors));
                }
                else {
                    setAnnotations(Collections.<CompilationError>emptyList());
                }
                
                SCLModuleEditorDocumentProvider.this.sourceViewer.updateCompletionAssistModuleName(moduleName);
            }
        });
    }

    protected void setAnnotations(List<CompilationError> errors) {
        synchronized(annotationModel.getLockObject()) {
            annotationModel.removeAllAnnotations();
            for(CompilationError error : errors) {
                Annotation annotation = new Annotation("org.eclipse.ui.workbench.texteditor.error", true,
                        error.description);
                int begin = Locations.beginOf(error.location);
                int end = Locations.endOf(error.location);
                Position position = new Position(begin, end - begin);
                annotationModel.addAnnotation(annotation, position);
            }
        }
    }

    boolean annotationsInitialized = false;

    @Override
    protected IAnnotationModel createAnnotationModel(Object element)
            throws CoreException {
        if(!annotationsInitialized) {
            updateAnnotations();
            annotationsInitialized = true;
        }
        return annotationModel;
    }

    @Override
    protected void doSaveDocument(IProgressMonitor monitor, Object element,
            IDocument document, boolean overwrite) throws CoreException {
        TimeLogger.resetTimeAndLog("SCLModuleEditorDocumentProvider.doSaveDocument");
        currentText = document.get();
        Simantics.getSession().asyncRequest(new WriteRequest() {
            @Override
            public void perform(WriteGraph graph) throws DatabaseException {
                graph.markUndoPoint();
                Layer0 L0 = Layer0.getInstance(graph);
                graph.claimLiteral(resource, L0.SCLModule_definition, currentText, Bindings.STRING);
                Layer0Utils.addCommentMetadata(graph, "Saved SCL Module " + graph.getRelatedValue2(resource, Layer0.getInstance(graph).HasName, Bindings.STRING));
            }
        });
    }

    @Override
    protected IRunnableContext getOperationRunner(IProgressMonitor monitor) {
        return null;
    }

    @Override
    public boolean isModifiable(Object element) {
        return !errorHappened && !immutable;
    }

    @Override
    public boolean isReadOnly(Object element) {
        return errorHappened || immutable;
    }

}
