/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.modeling.ui.actions.style;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Point;

import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.simantics.Simantics;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.Session;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.request.ReadRequest;
import org.simantics.db.common.request.WriteRequest;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.G2DUtils;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.stubs.G2DResource;
import org.simantics.modeling.ui.Activator;
import org.simantics.ui.SimanticsUI;
import org.simantics.utils.strings.format.MetricsFormat;
import org.simantics.utils.ui.ErrorLogger;

/**
 * Style Edit
 * 
 * TODO : should we have extension point for expanding styles?
 * TODO : default ColorChooser is not suitable for this task
 * TODO : how to store MetricsFormat template list
 * 
 * @author Marko Luukkainen <marko.luukkainen@vtt.fi>
 */
public class EditStyle {

    private static final String SECTION_AWT_STYLE_DIALOG = "AWTStyleDialog";
    private static final String SETTING_DIALOG_HEIGHT    = "h";
    private static final String SETTING_DIALOG_WIDTH     = "w";
    private static final String SETTING_DIALOG_Y         = "y";
    private static final String SETTING_DIALOG_X         = "x";

    public static void openStyleDialog(final Resource[] resources) {
        if (resources.length == 0)
            return;

        Simantics.getSession().asyncRequest(new ReadRequest() {

            @Override
            public void run(ReadGraph graph) throws DatabaseException {
                G2DResource g2d = G2DResource.getInstance(graph);
                DiagramResource dr = DiagramResource.getInstance(graph);
                boolean hasStyle = true;
                Color color = null;
                Font font = null;
                MetricsFormat format = null;

                boolean hasColor = true;
                boolean hasFont = true;
                boolean hasFormat = true;

                // Find what kind of styles selected objects support
                for (Resource r : resources) {
                    if (!graph.isInstanceOf(r, dr.StyleProvider)) {
                        hasStyle = false;
                        break;
                    }
                    if (graph.isInstanceOf(r, dr.FontProvider)) {
                        if (font == null) {
                            Resource fontR = graph.getPossibleObject(r, g2d.HasFont);
                            if (fontR != null)
                                font = G2DUtils.getFont(graph,fontR);
                        }
                    } else {
                        hasFont = false;
                    }
                    if (graph.isInstanceOf(r, dr.ColorProvider)) {
                        if (color == null) {
                            Resource colorR = graph.getPossibleObject(r, g2d.HasColor);
                            if (colorR != null)
                                color = G2DUtils.getColor(graph,colorR);
                        }
                    } else {
                        hasColor = false;
                    }
                    if (graph.isInstanceOf(r, dr.FormatProvider)) {
                        if (format == null) {
                            Resource formatR = graph.getPossibleObject(r, dr.HasFormat);
                            if (formatR != null)
                                format = G2DUtils.getMetricsFormat(graph,formatR);
                        }
                    } else {
                        hasFormat = false;
                    }
                }

                if (!hasStyle)
                    return; // TODO : show error

                if (!hasFont && !hasColor && !hasFormat)
                    return; // TODO : show error

                final Font currentFont = font;
                final Color currentColor = color;
                final MetricsFormat currentFormat = format;
                final boolean useFont = hasFont;
                final boolean useColor = hasColor;
                final boolean useFormat = hasFormat;

                Job job = new Job("Open Style Dialog") {
                    @Override
                    protected IStatus run(IProgressMonitor monitor) {
                        monitor.beginTask("Open dialog", IProgressMonitor.UNKNOWN);
                        SwingUtilities.invokeLater(new Runnable() {
                            @Override
                            public void run() {
                                setThread(Thread.currentThread());
                                try {
                                    UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
                                    // there's no reason to show errors to user.
                                } catch (ClassNotFoundException e) {
                                    ErrorLogger.defaultLogError(e);
                                } catch (InstantiationException e) {
                                    ErrorLogger.defaultLogError(e);
                                } catch (IllegalAccessException e) {
                                    ErrorLogger.defaultLogError(e);
                                } catch (UnsupportedLookAndFeelException e) {
                                    ErrorLogger.defaultLogError(e);
                                }

                                AWTStyleDialog dialog = new AWTStyleDialog(useFont, useColor, useFormat);
                                if (useColor && currentColor != null)
                                    dialog.setStartColor(currentColor);
                                if (useFont && currentFont != null)
                                    dialog.setStartFont(currentFont);
                                if (useFormat && currentFormat != null) {
                                    dialog.setStartFormat(currentFormat);
                                }

                                // Restore dialog settings
                                IDialogSettings ds = Activator.getDefault().getDialogSettings();
                                IDialogSettings sd = ds.getSection(SECTION_AWT_STYLE_DIALOG);
                                boolean restoredSettings = false;
                                if (sd == null) {
                                    sd = ds.addNewSection(SECTION_AWT_STYLE_DIALOG);
                                } else {
                                    try {
                                        int x = sd.getInt(SETTING_DIALOG_X);
                                        int y = sd.getInt(SETTING_DIALOG_Y);
                                        int w = sd.getInt(SETTING_DIALOG_WIDTH);
                                        int h = sd.getInt(SETTING_DIALOG_HEIGHT);
                                        // Sanity check
                                        if (w > 0 && h > 0) {
                                            dialog.setLocation(x, y);
                                            dialog.setSize(w, h);
                                            restoredSettings = true;
                                        }
                                    } catch (NumberFormatException e) {
                                        // Ignore.
                                    }
                                }
                                if (!restoredSettings) {
                                    dialog.setLocationByPlatform(true);
                                    dialog.pack();
                                }
                                done(Status.OK_STATUS);
                                dialog.setVisible(true);

                                // Save settings
                                Point loc = dialog.getLocation();
                                Dimension dim = dialog.getSize();
                                sd.put(SETTING_DIALOG_X, loc.x);
                                sd.put(SETTING_DIALOG_Y, loc.y);
                                sd.put(SETTING_DIALOG_WIDTH, dim.width);
                                sd.put(SETTING_DIALOG_HEIGHT, dim.height);

                                if (!dialog.isCancelled()) {
                                    // OK was pressed
                                    final Font font = dialog.getFont();
                                    final Color color = dialog.getColor();
                                    final MetricsFormat format = dialog.getFormat();
                                    Session session = Simantics.getSession();
                                    session.markUndoPoint();
                                    session.asyncRequest(new WriteRequest() {

                                        @Override
                                        public void perform(WriteGraph graph) throws DatabaseException {
                                            G2DResource g2d = G2DResource.getInstance(graph);
                                            DiagramResource dr = DiagramResource.getInstance(graph);
                                            // create style definitions
                                            Resource fontResource = null;
                                            if (useFont && font != null)
                                                fontResource = G2DUtils.createFont(graph, font);

                                            Resource colorResource = null;
                                            if (useColor && color != null)
                                                colorResource = G2DUtils.createColor(graph, color);

                                            Resource formatResource = null;
                                            if (useFormat)
                                                formatResource = G2DUtils.createMetricsFormat(graph, format);

                                            // use style definitions in selected objects
                                            for (Resource r : resources) {
                                                if (useFont && fontResource != null && graph.isInstanceOf(r, dr.FontProvider)) {
                                                    graph.deny(r, g2d.HasFont);
                                                    graph.claim(r, g2d.HasFont, fontResource);
                                                }
                                                if (useColor && colorResource != null && graph.isInstanceOf(r, dr.ColorProvider)) {
                                                    graph.deny(r, g2d.HasColor);
                                                    graph.claim(r, g2d.HasColor, colorResource);
                                                }
                                                if (useFormat && formatResource != null && graph.isInstanceOf(r, dr.FormatProvider)) {
                                                    graph.deny(r,dr.HasFormat);
                                                    graph.claim(r, dr.HasFormat, formatResource);
                                                }
                                            }
                                        }
                                    });
                                }
                            }
                        });
                        return Job.ASYNC_FINISH;
                    }
                };
                job.setUser(true);
                job.schedule();
            }
        });
    }

}
