package org.simantics.modeling.ui.actions;

import java.util.ArrayList;
import java.util.Iterator;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTableViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ICheckStateProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.SelectionDialog;
import org.eclipse.ui.internal.IWorkbenchHelpContextIds;
import org.eclipse.ui.internal.WorkbenchMessages;
import org.simantics.ui.internal.Activator;

@SuppressWarnings("restriction")
public abstract class AssignSymbolGroupsDialog extends SelectionDialog {

    private static final String DIALOG = "AssignSymbolGroupsDialog"; //$NON-NLS-1$

    static String SELECT_ALL_TITLE = WorkbenchMessages.SelectionDialog_selectLabel;

    static String DESELECT_ALL_TITLE = WorkbenchMessages.SelectionDialog_deselectLabel;

    // the root element to populate the viewer with
    protected Object inputElement;

    // providers for populating this dialog
    private ILabelProvider labelProvider;

    private IStructuredContentProvider contentProvider;

    private ICheckStateProvider checkStateProvider;

    // the visual selection widget group
    protected CheckboxTableViewer listViewer;

    private IDialogSettings     dialogBoundsSettings;

    // sizing constants
    private final static int SIZING_SELECTION_WIDGET_HEIGHT = 250;

    private final static int SIZING_SELECTION_WIDGET_WIDTH = 300;

    /**
     * Creates a list selection dialog.
     *
     * @param parentShell the parent shell
     * @param input	the root element to populate this dialog with
     * @param contentProvider the content provider for navigating the model
     * @param labelProvider the label provider for displaying model elements
     * @param message the message to be displayed at the top of this dialog, or
     *    <code>null</code> to display a default message
     */
    public AssignSymbolGroupsDialog(Shell parentShell, Object input,
            IStructuredContentProvider contentProvider,
            ILabelProvider labelProvider,
            ICheckStateProvider checkStateProvider,
            String message) {
        super(parentShell);
        setTitle(WorkbenchMessages.ListSelection_title);
        inputElement = input;
        this.contentProvider = contentProvider;
        this.labelProvider = labelProvider;
        this.checkStateProvider = checkStateProvider;
        if (message != null) {
            setMessage(message);
        } else {
            setMessage(WorkbenchMessages.ListSelection_message);
        }

        IDialogSettings settings = Activator.getDefault().getDialogSettings();
        dialogBoundsSettings = settings.getSection(DIALOG);
        if (dialogBoundsSettings == null)
            dialogBoundsSettings = settings.addNewSection(DIALOG);
    }

    @Override
    protected IDialogSettings getDialogBoundsSettings() {
        return dialogBoundsSettings;
    }

    /**
     * Add the selection and deselection buttons to the dialog.
     * @param composite org.eclipse.swt.widgets.Composite
     */
    private void addSelectionButtons(Composite composite) {
        Composite buttonComposite = new Composite(composite, SWT.NONE);
        GridLayout layout = new GridLayout();        
		layout.marginWidth = 0;
		layout.horizontalSpacing = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_SPACING);
		layout.verticalSpacing = convertHorizontalDLUsToPixels(IDialogConstants.VERTICAL_SPACING);
        buttonComposite.setLayout(layout);
        buttonComposite.setLayoutData(new GridData(SWT.LEFT, SWT.TOP, false, true));

        Button selectButton = createButton(buttonComposite,
                IDialogConstants.SELECT_ALL_ID, SELECT_ALL_TITLE, false);

        SelectionListener listener = new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                listViewer.setAllChecked(true);
                checkStateChanged(contentProvider.getElements(inputElement), true);
            }
        };
        selectButton.addSelectionListener(listener);

        Button deselectButton = createButton(buttonComposite,
                IDialogConstants.DESELECT_ALL_ID, DESELECT_ALL_TITLE, false);

        listener = new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                listViewer.setAllChecked(false);
                checkStateChanged(contentProvider.getElements(inputElement), false);
            }
        };
        deselectButton.addSelectionListener(listener);

        @SuppressWarnings("unused")
        Label label = new Label(buttonComposite, SWT.NONE);

        Button newButton = createButton(buttonComposite,
                IDialogConstants.INTERNAL_ID-1, "&New...", false);

        listener = new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                newAction();
            }
        };
        newButton.addSelectionListener(listener);

        Button removeButton = createButton(buttonComposite,
                IDialogConstants.INTERNAL_ID-2, "&Remove", false);

        listener = new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                deleteAction(
                        ((IStructuredSelection)listViewer.getSelection())
                        .toArray());
            }
        };
        removeButton.addSelectionListener(listener);

        layout.numColumns = 1;
    }
    


    protected abstract void newAction();

    protected abstract void deleteAction(Object[] array);

    protected abstract void checkStateChanged(Object[] elements, boolean checked);

    /**
     * Visually checks the previously-specified elements in this dialog's list 
     * viewer.
     */
    private void checkInitialSelections() {
        Iterator<?> itemsToCheck = getInitialElementSelections().iterator();

        while (itemsToCheck.hasNext()) {
			listViewer.setChecked(itemsToCheck.next(), true);
		}
    }

    /*
     *  (non-Javadoc)
     * @see org.eclipse.jface.window.Window#configureShell(org.eclipse.swt.widgets.Shell)
     */
    protected void configureShell(Shell shell) {
        super.configureShell(shell);
        PlatformUI.getWorkbench().getHelpSystem().setHelp(shell,
				IWorkbenchHelpContextIds.LIST_SELECTION_DIALOG);
    }

    /* (non-Javadoc)
     * Method declared on Dialog.
     */
    protected Control createDialogArea(Composite parent) {
        // page group
        Composite composite = (Composite) super.createDialogArea(parent);
        
        initializeDialogUnits(composite);
        
        Composite cc = new Composite(composite, SWT.NONE);
        {
            GridLayout layout = new GridLayout();
            layout.marginHeight = 0;
            layout.marginWidth = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_MARGIN);
            layout.verticalSpacing = convertVerticalDLUsToPixels(IDialogConstants.VERTICAL_SPACING);
            layout.horizontalSpacing = convertHorizontalDLUsToPixels(IDialogConstants.HORIZONTAL_SPACING);
            layout.numColumns = 2;
            cc.setLayout(layout);
            cc.setLayoutData(new GridData(GridData.FILL_BOTH));
        }
        
        Label label = createMessageArea(cc);
        GridDataFactory.fillDefaults().grab(true, false).span(2, 1).applyTo(label);
        
        listViewer = CheckboxTableViewer.newCheckList(cc, SWT.BORDER | SWT.MULTI | SWT.FULL_SELECTION);
        GridData data = new GridData(GridData.FILL_BOTH);
        data.heightHint = SIZING_SELECTION_WIDGET_HEIGHT;
        data.widthHint = SIZING_SELECTION_WIDGET_WIDTH;
        listViewer.getTable().setLayoutData(data);

        listViewer.setLabelProvider(labelProvider);
        listViewer.setContentProvider(contentProvider);
        listViewer.setCheckStateProvider(checkStateProvider);
        listViewer.getTable().addKeyListener(new KeyAdapter() {
            @Override
            public void keyPressed(KeyEvent e) {
                if(e.keyCode == SWT.DEL) {
                    deleteAction(
                            ((IStructuredSelection)listViewer.getSelection())
                            .toArray());
                }
            }
        });
        listViewer.addCheckStateListener(new ICheckStateListener() {
            @Override
            public void checkStateChanged(CheckStateChangedEvent event) {
                AssignSymbolGroupsDialog.this.checkStateChanged(new Object[] { event.getElement() }, event.getChecked());
            }
        });

        addSelectionButtons(cc);

        initializeViewer();

        // initialize page
        if (!getInitialElementSelections().isEmpty()) {
			checkInitialSelections();
		}

        Dialog.applyDialogFont(composite);
        
        return composite;
    }

    /**
     * Returns the viewer used to show the list.
     * 
     * @return the viewer, or <code>null</code> if not yet created
     */
    protected CheckboxTableViewer getViewer() {
        return listViewer;
    }

    /**
     * Initializes this dialog's viewer after it has been laid out.
     */
    private void initializeViewer() {
        listViewer.setInput(inputElement);
    }

    /**
     * The <code>ListSelectionDialog</code> implementation of this 
     * <code>Dialog</code> method builds a list of the selected elements for later
     * retrieval by the client and closes this dialog.
     */
    protected void okPressed() {

        // Get the input children.
        Object[] children = contentProvider.getElements(inputElement);

        // Build a list of selected children.
        if (children != null) {
            ArrayList<Object> list = new ArrayList<Object>();
            for (int i = 0; i < children.length; ++i) {
                Object element = children[i];
                if (listViewer.getChecked(element)) {
					list.add(element);
				}
            }
            setResult(list);
        }

        super.okPressed();
    }

}
