/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.graphviz.ui;

import java.awt.Canvas;
import java.awt.Frame;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.swing.SwingUtilities;

import org.eclipse.swt.SWT;
import org.eclipse.swt.awt.SWT_AWT;
import org.eclipse.swt.widgets.Composite;
import org.simantics.graphviz.Graph;
import org.simantics.graphviz.continuation.Computation;
import org.simantics.graphviz.continuation.Continuation;
import org.simantics.graphviz.drawable.GraphDrawable2;
import org.simantics.graphviz.drawable.ViewerCanvas;

public class GraphvizComponent2 extends Composite {

    Canvas canvas;
    GraphDrawable2 drawable;
    Graph graph;
    AtomicBoolean initialized = new AtomicBoolean(false);
    Frame frame;

    public GraphvizComponent2(Composite parent, int style) {
        super(parent, style | SWT.EMBEDDED | SWT.NO_BACKGROUND);

        frame = SWT_AWT.new_Frame(this);

        SwingUtilities.invokeLater(new Runnable() {

            @Override
            public void run() {
                try {
                    drawable = new GraphDrawable2();
                    canvas = new ViewerCanvas(drawable);
                    frame.add(canvas);
                } finally {
                    synchronized (initialized) {
                        initialized.set(true);
                        initialized.notifyAll();
                    }
                }
            }

        });
    }

    public void waitUntilInitialized() {
        try {
            synchronized (initialized) {
                while (!initialized.get())
                    initialized.wait();
            }
        } catch (InterruptedException e) {
            throw new Error("GraphvizComponent AWT population interrupted for class " + this, e);
        }
    }

    /**
     * Sets a new graph to be drawn. This operation may take a while. It can
     * be called from any thread. Automatically redraws the component.
     * @param graph
     */
    public Computation<Graph> setGraph(Graph graph) {
        return setGraph(graph, "dot");
    }

    /**
     * Sets a new graph to be drawn. This operation may take a while. It can
     * be called from any thread. Automatically redraws the component.
     * @param graph
     */
    public Computation<Graph> setGraph(Graph graph, String algorithm) {
        Computation<Graph> computation = drawable.setGraph(graph, algorithm);
        computation.addContinuation(new Continuation<Graph>() {
            
            @Override
            public void succeeded(Graph result) {
                if (isDisposed())
                    return;
                
                fit();
            }
            
            @Override
            public void failed(Exception exception) {
            }
        });
        return computation;
    }

    /**
     * Fits the content of the canvas to the component.
     * Can be called from any thread.
     */
    public void fit() {
        ((ViewerCanvas)canvas).fit();
        getDisplay().asyncExec(new Runnable() {

            @Override
            public void run() {
                redraw();
            }

        });
        SwingUtilities.invokeLater(new Runnable() {

            @Override
            public void run() {
                canvas.repaint();
            }

        });


    }

    public void requestFocus() {
        if(canvas != null)
            canvas.requestFocus();
    }
    
    public Frame getFrame() {
    	return frame;
    }
    
    public Canvas getCanvas() {
    	return canvas;
    }
    
    public GraphDrawable2 getDrawable() {
    	return drawable;
    }

}
