/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.utils;

import java.io.Serializable;

import org.simantics.scenegraph.utils.GridUtils;


/**
 * Grid configuration.
 * 
 * @author Tuukka Lehtonen
 * @author Marko Luukkainen
 * @author Toni Kalajainen
 */
public final class GridSpacing implements Serializable {

    private static final long serialVersionUID = 8445639004963828463L;

    /** When the view scale gets beneath this value a grid should no longer be shown */
    public static final double GRID_MIN_USER_SIZE = 1e-6;

	public static final GridSpacing SOME_SPACING = GridSpacing.makeGridSpacing(100, 100, 40);
    
    /** The minimum amount of pixel between major grid lines. */
    public double     	minPixels;

    /** Grid segment sizes (canvas coordinates) */
    public double		segment, segmentExp;

    /** Conversions between canvas (unit) and control (pixel) */
    public double		pixelsPerUnit;
    public double     	unitsPerPixel;

    public double		pixelsPerSegment;
    
    public GridSpacing(double spacing, double spacingExp, double unitsPerPixel, double minPixels)
    {
        this.unitsPerPixel = unitsPerPixel;
        pixelsPerUnit = 1 / unitsPerPixel;
        this.segment = spacing; 
        this.segmentExp = spacingExp;
        pixelsPerSegment = spacing * pixelsPerUnit;
        this.minPixels = minPixels;
    }

    @Override
    public boolean equals(Object obj) {
    	if (obj == null) return false;
    	if (obj == this) return true;
        if (obj instanceof GridSpacing == false) return false;
        GridSpacing g2 = (GridSpacing)obj;
        if(pixelsPerUnit != g2.pixelsPerUnit) return false;
        if(pixelsPerSegment != g2.pixelsPerSegment) return false;
        if(unitsPerPixel != g2.unitsPerPixel) return false;
        if(segment != g2.segment) return false;
        if(segmentExp != g2.segmentExp) return false;
        return true;
    }

    /**
     * Calculates grid cofiguration
     * 
     * @return grid configuration 
     */
    public static GridSpacing makeGridSpacing(
    		double viewboxSize,  
    		double controlWidth,  
    		double minPixels)
    {
        if (controlWidth==0) controlWidth = 1;
        if (Math.abs(viewboxSize) < GRID_MIN_USER_SIZE) viewboxSize = GRID_MIN_USER_SIZE * Math.signum(viewboxSize);

        double unitsPerPixel = viewboxSize / Math.max(controlWidth, minPixels);
        double [] gridSpacing = getGridSpacing(unitsPerPixel, minPixels);
        return new GridSpacing(gridSpacing[0], gridSpacing[1], unitsPerPixel, minPixels);
    }

    public static GridSpacing makeGridSpacingForTime(
    		double viewboxSize,  
    		double controlWidth,  
    		double minPixels)
    {
        if (controlWidth==0) controlWidth = 1;
        if (viewboxSize < GRID_MIN_USER_SIZE) viewboxSize = 1;

        double unitsPerPixel = viewboxSize / Math.max(controlWidth, minPixels);
        double [] gridSpacing = getGridSpacing(unitsPerPixel, minPixels);
        return new GridSpacing(gridSpacing[0], gridSpacing[1], unitsPerPixel, minPixels);
    }
    
    private static double[] getGridSpacing(double unitsPerPixel, double minPixels) {
        double minGridSize = minPixels * unitsPerPixel;
        double[] gridSpacing = GridUtils.getEvenGridSpacingWithExp(minGridSize, new double[2]);
        return gridSpacing;
    }

    /**
     * Snap point to the grid. The snapping is based on grid last paint operation.
     * 
     * @param pos
     * @return
     */
    public double snapToGrid(double pos) {
        return GridUtils.snapToGrid(pos, segment);
    }

}
