/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.g2d.elementclass;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Collection;

import org.simantics.g2d.diagram.handler.Topology.Terminal;
import org.simantics.g2d.element.ElementClass;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;
import org.simantics.g2d.element.handler.InternalSize;
import org.simantics.g2d.element.handler.SceneGraph;
import org.simantics.g2d.element.handler.TerminalLayout;
import org.simantics.g2d.element.handler.TerminalTopology;
import org.simantics.g2d.element.handler.impl.BorderColorImpl;
import org.simantics.g2d.element.handler.impl.DefaultTransform;
import org.simantics.g2d.element.handler.impl.FillColorImpl;
import org.simantics.g2d.element.handler.impl.Resizeable;
import org.simantics.g2d.element.handler.impl.TextImpl;
import org.simantics.g2d.element.handler.impl.TextPainter;
import org.simantics.g2d.utils.geom.DirectionSet;
import org.simantics.scenegraph.g2d.G2DParentNode;
import org.simantics.scenegraph.g2d.nodes.ShapeNode;
import org.simantics.utils.datastructures.hints.IHintContext.KeyOf;

/**
 * 
 * @author Toni Kalajainen
 */
public class BoxClass  {

    public static final ElementClass BOXCLASS =
        ElementClass.compile(
                DefaultTransform.INSTANCE,
                Resizeable.UNCONSTRICTED,
                BorderColorImpl.BLACK,
                FillColorImpl.WHITE,
                TextImpl.INSTANCE,
                new RectangleSGNode(),
                new BoxNode(),
                new TextPainter()
                //new RotatorHandler()
        );

    final static BasicStroke STROKE = new BasicStroke(1.0f);

    static class RectangleSGNode implements SceneGraph {
        /**
         * 
         */
        private static final long serialVersionUID = -1550908516884986157L;
        private G2DParentNode node = null;

        @Override
        public void cleanup(IElement e) {
            if(node != null) {
                node.remove();
            }
            node = null;
        }

        @Override
        public void init(IElement e, G2DParentNode parent) {
            if(node == null) {
                node = parent.addNode(G2DParentNode.class);
            }
            ShapeNode bgnode = node.getOrCreateNode("bg", ShapeNode.class);
            bgnode.setZIndex(1);
            ShapeNode fgnode = node.getOrCreateNode("fg", ShapeNode.class);
            fgnode.setZIndex(2);

            Color fc = ElementUtils.getFillColor(e);
            Color bc = ElementUtils.getBorderColor(e);

            Rectangle2D rect = ElementUtils.getElementBounds(e);
            bgnode.setColor(fc);
            bgnode.setFill(true);
            bgnode.setShape(rect);

            fgnode.setColor(bc);
            fgnode.setFill(false);
            fgnode.setStroke(STROKE);
            fgnode.setScaleStroke(true);
            fgnode.setShape(rect);
        }
    }



    static class TerminalPoint implements Terminal {
    }

    public static class TerminalKeyOf extends KeyOf {
        public final Terminal terminal;

        private final int hashCode;

        public TerminalKeyOf(Terminal terminal, Class<?> clazz) {
            super(clazz);
            this.terminal = terminal;
            hashCode = terminal.hashCode() ^ clazz.hashCode();
        }

        @Override
        public int hashCode() {
            return hashCode;
        }

        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof TerminalKeyOf)) return false;
            TerminalKeyOf other = (TerminalKeyOf) obj;
            if (!other.terminal.equals(terminal)) return false;
            return super.equals(obj);
        }
    }

    public static class TerminalConnectKey extends TerminalKeyOf {
        public TerminalConnectKey(Terminal terminal) {
            super(terminal, IElement.class);
        }

    }

    static class BoxNode implements TerminalTopology, TerminalLayout {
        private static final long serialVersionUID = 4561446983606561224L;
        public static final Terminal T0 = new TerminalPoint();
        public static final Terminal T1 = new TerminalPoint();
        public static final Terminal T2 = new TerminalPoint();
        public static final Terminal T3 = new TerminalPoint();
        public static final Terminal T4 = new TerminalPoint();
        public static final Terminal T5 = new TerminalPoint();
        public static final Terminal T6 = new TerminalPoint();
        public static final Terminal T7 = new TerminalPoint();
        public static final Terminal T8 = new TerminalPoint();

        @Override
        public void getTerminals(IElement e, Collection<Terminal> result) {
            result.add(T0);
            result.add(T1);
            result.add(T2);
            result.add(T3);
            result.add(T4);
            result.add(T5);
            result.add(T6);
            result.add(T7);
            result.add(T8);
        }

        @Override
        public AffineTransform getTerminalPosition(IElement node, Terminal t)
        {
            InternalSize b = node.getElementClass().getSingleItem(InternalSize.class);
            Rectangle2D rect = b.getBounds(node, null);
            Point2D pos = null;
            if (rect == null) {
                pos = new Point2D.Double(0, 0);
            } else {
                if (t==T0) pos = new Point2D.Double(rect.getCenterX(), rect.getCenterY());
                if (t==T1) pos = new Point2D.Double(rect.getMinX(), rect.getMinY());
                if (t==T2) pos = new Point2D.Double(rect.getCenterX(), rect.getMinY());
                if (t==T3) pos = new Point2D.Double(rect.getMaxX(), rect.getMinY());
                if (t==T4) pos = new Point2D.Double(rect.getMaxX(), rect.getCenterY());
                if (t==T5) pos = new Point2D.Double(rect.getMaxX(), rect.getMaxY());
                if (t==T6) pos = new Point2D.Double(rect.getCenterX(), rect.getMaxY());
                if (t==T7) pos = new Point2D.Double(rect.getMinX(), rect.getMaxY());
                if (t==T8) pos = new Point2D.Double(rect.getMinX(), rect.getCenterY());
            }
            AffineTransform result = new AffineTransform();
            result.setToTranslation(pos.getX(), pos.getY());
            return result;
        }

        @Override
        public boolean getTerminalDirection(IElement node, Terminal t, DirectionSet directions) {
            if (t==T0) {
                directions.addAll(DirectionSet.NESW);
                return true;
            }
            if (t==T1) {
                directions.addAll(DirectionSet.NW2);
                return true;
            }
            if (t==T2) {
                directions.addAll(DirectionSet.N);
                return true;
            }
            if (t==T3) {
                directions.addAll(DirectionSet.NE2);
                return true;
            }
            if (t==T4) {
                directions.addAll(DirectionSet.E);
                return true;
            }
            if (t==T5) {
                directions.addAll(DirectionSet.SE2);
                return true;
            }
            if (t==T6) {
                directions.addAll(DirectionSet.S);
                return true;
            }
            if (t==T7) {
                directions.addAll(DirectionSet.SW2);
                return true;
            }
            if (t==T8) {
                directions.addAll(DirectionSet.W);
                return true;
            }
            return false;
        }

        public static final Shape BOX_SHAPE = new Rectangle(-3, -3, 7, 7);
        public static final Shape CIRCLE_SHAPE = new Ellipse2D.Double(-5, -5, 9, 9);

        @Override
        public Shape getTerminalShape(IElement node, Terminal t) {
            return (t==T6 || t==T7 || t==T0) ? CIRCLE_SHAPE : null;
        }

    }

}
