/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.synchronization.graph;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.utils.OrderedSetUtils;
import org.simantics.diagram.synchronization.ModificationAdapter;
import org.simantics.diagram.ui.DiagramModelHints;
import org.simantics.g2d.diagram.IDiagram;
import org.simantics.g2d.element.ElementUtils;
import org.simantics.g2d.element.IElement;

/**
 * This modification reorders the specified diagram in the graph backend
 * according to the order of the elements specified argument element list.
 * 
 * <p>
 * The algorithm is linear with respect to the amount of elements on the
 * diagram. It tries to minimize the number of ordered set operations.
 * 
 * @author Tuukka Lehtonen
 */
public class ElementReorder extends ModificationAdapter {

    IDiagram diagram;
    List<IElement> order;

    public ElementReorder(IDiagram diagram, List<IElement> order) {
        super(LOW_PRIORITY);
        this.diagram = diagram;
        this.order = order;
    }

    @Override
    public void perform(WriteGraph g) throws Exception {
        Resource l = diagram.getHint(DiagramModelHints.KEY_DIAGRAM_RESOURCE);

        List<Resource> graphOrder = OrderedSetUtils.toList(g, l);
        Set<Resource> graphContents = new HashSet<Resource>(graphOrder);

        List<Resource> diagramOrder = new ArrayList<Resource>(order.size());
        Map<Resource, Integer> diagramOrderIndex = new HashMap<Resource, Integer>(order.size());
        int i = 0;
        for (IElement e : order) {
            Object obj = ElementUtils.getObject(e);
            if (obj instanceof Resource) {
                Resource r = (Resource) obj;
                // Only consider resources that still are in the diagram.
                // This prevents errors in situations where #order contains
                // elements that no longer exist in the diagram.
                if (graphContents.contains(r)) {
                    diagramOrder.add(r);
                    diagramOrderIndex.put(r, Integer.valueOf(i));
                    ++i;
                }
            }
        }

        // Reorder the backend list according to diagramOrder
        i = 0;
        for (Resource r : graphOrder) {
            Integer di = diagramOrderIndex.get(r);
            if (di != null) {
                int targetIndex = di;
                int graphIndex = i++;
                if (graphIndex != targetIndex) {
                    // Check if the predecessor of r is already correct.
                    // If it is, we don't have to do anything for r.
                    Resource graphPrev = OrderedSetUtils.prev(g, l, r);
                    Resource after = null;
                    if (targetIndex == 0) {
                        after = l;
                        if (l.equals(graphPrev))
                            continue;
                    } else {
                        after = diagramOrder.get(targetIndex - 1);
                        if (after.equals(graphPrev))
                            continue;
                    }

                    // r needs to be repositioned.
                    OrderedSetUtils.remove(g, l, r);
                    OrderedSetUtils.addAfter(g, l, after, r);
                }
            }
        }
    }

}
