/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.synchronization;

import org.simantics.utils.threads.IThreadWorkQueue;

/**
 * An abstraction for a queue of modifications to be performed into a backend
 * model for synchronization with the diagram model.
 * 
 * @author Tuukka Lehtonen
 */
public interface IModificationQueue {

    /**
     * Asynchronously perform the specified modification, using the specified
     * thread for the modification completion event. This method will <b>not</b>
     * cause any write request invocations to the backend, it merely offers the
     * modifications up for execution. After several modifications have been
     * offered, {@link #flush()} may be invoked to execute the offered
     * modifications to execute multiple modifications in one transaction.
     * 
     * <p>
     * This method should be equal to calling
     * {@link #offer(IModification, IThreadWorkQueue)} with a null
     * IThreadWorkQueue argument.
     * 
     * @param m the modification to execute
     * @return <code>true</code> if the modification was added to the queue
     */
    boolean offer(IModification m);

    /**
     * Asynchronously perform the specified modification, using the specified
     * thread for the modification completion event. This method will <b>not</b>
     * cause any write request invocations to the backend, it merely offers the
     * modifications up for execution. After several modifications have been
     * offered, {@link #flush()} may be invoked to execute the offered
     * modifications to execute multiple modifications in one transaction.
     * 
     * @param m the modification to execute
     * @param completionThread the thread to be used for running the
     *        modification completion event or <code>null</code> to not care
     *        about the thread.
     * @return <code>true</code> if the modification was added to the queue
     */
    boolean offer(IModification m, IThreadWorkQueue completionThread);

    /**
     * Flushes the queue of modifications. All modifications shall be dispatched
     * to be executed but this method can return at any time before the
     * modifications have been performed.
     */
    void flush();

    /**
     * Finishes the queue of modifications gathered so far. This method will
     * block until all current modifications have been ran to completion.
     * 
     * @throws InterruptedException if the waiting thread is interrupted
     */
    void finish() throws InterruptedException;

}
