/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.symbolcontribution;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.request.BinaryRead;
import org.simantics.db.common.request.ObjectsWithSupertype;
import org.simantics.db.common.utils.NameUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.diagram.symbollibrary.ISymbolGroup;
import org.simantics.diagram.symbollibrary.ISymbolItem;
import org.simantics.layer0.Layer0;

/**
 * A basic SymbolProviderFactory implementation for the graph database. It is
 * capable of loading an ISymbolProvider from BasicSymbolContribution instances
 * like the following:
 * 
 * <pre>
 * _ : DIA.BasicSymbolContribution
 *     DIA.BasicSymbolContributionHasSymbolLibrary
 *         MySymbolLibrary1
 *         MySymbolLibrary2
 * </pre>
 * 
 * @author Tuukka Lehtonen
 */
public class BasicSymbolProviderFactory implements SymbolProviderFactory {
		
    Resource contribution;
    Resource diagram;

    public BasicSymbolProviderFactory(Resource contribution, Resource diagram) {
        this.contribution = contribution;
        this.diagram = diagram;
    }

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((contribution == null) ? 0 : contribution.hashCode());
		result = prime * result + ((diagram == null) ? 0 : diagram.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		BasicSymbolProviderFactory other = (BasicSymbolProviderFactory) obj;
		if (contribution == null) {
			if (other.contribution != null)
				return false;
		} else if (!contribution.equals(other.contribution))
			return false;
		if (diagram == null) {
			if (other.diagram != null)
				return false;
		} else if (!diagram.equals(other.diagram))
			return false;
		return true;
	}
    
    @Override
    public ISymbolProvider create(ReadGraph g) throws DatabaseException {
        return g.syncRequest(new LoadRequest(contribution, diagram));
    }

    /*
     * Note: this cannot be ResourceRead since it must never be
     * classified as immutable because of possible dynamic filters
     */
    static class LoadRequest extends BinaryRead<Resource, Resource, ISymbolProvider> {
        public LoadRequest(Resource contribution, Resource diagram) {
            super(contribution, diagram);
        }
        @Override
        public ISymbolProvider perform(ReadGraph graph) throws DatabaseException {
            Layer0 l0 = Layer0.getInstance(graph);
            DiagramResource dr = DiagramResource.getInstance(graph);

            if (!SymbolProviderFactories.accept(graph, dr, parameter, parameter2))
                return new SymbolProvider(Collections.<ISymbolGroup>emptyList());

            Collection<Resource> libraries = graph.getObjects(parameter, dr.BasicSymbolContributionHasSymbolLibrary);
            Collection<ISymbolGroup> groups = new ArrayList<ISymbolGroup>(libraries.size());

            for (Resource library : libraries) {
                if (SymbolProviderFactories.accept(graph, dr, library, parameter2)) {
                    groups.add(createGroup(graph, library, l0.DependsOn, parameter2));
                }
            }

            return new SymbolProvider(groups);
            
        }
    }

    static class SymbolProvider extends AbstractSymbolProvider {
        public SymbolProvider(Collection<ISymbolGroup> groups) {
            super();
            setSymbolGroup(groups);
            lockGroups();
        }
    }

    static SymbolGroup createGroup(ReadGraph graph, Resource library, Resource relation, Resource diagram) throws DatabaseException {
        Layer0 l0 = Layer0.getInstance(graph);
        DiagramResource dr = DiagramResource.getInstance(graph);

        String name = safeName(graph, library);
        String description = graph.getPossibleRelatedValue2(library, l0.HasDescription, Bindings.STRING);
        if (description == null)
            description = name;
        ModifiableSymbolGroup group = new ModifiableSymbolGroup(library, name, description);

        ArrayList<ISymbolItem> items = new ArrayList<ISymbolItem>();
        for (Resource item : graph.syncRequest(new ObjectsWithSupertype(library, relation, dr.Element))) {
            if (!SymbolProviderFactories.accept(graph, dr, item, diagram)) {
                continue;
            }

            String itemName = safeName(graph, item);
            String itemDescription = graph.getPossibleRelatedValue2(item, l0.HasDescription, Bindings.STRING);
            if (itemDescription == null || itemDescription.isEmpty())
                itemDescription = itemName;
            items.add( new ElementSymbolItem(item, itemName, itemDescription, group) );
        }

        Collections.sort(items, new Comparator<ISymbolItem>() {
            @Override
            public int compare(ISymbolItem o1, ISymbolItem o2) {
                return o1.getName().compareToIgnoreCase(o2.getName());
            }
        });

        group.setItems(items.toArray(new ISymbolItem[items.size()]));

        return group;
    }

    private static String safeName(ReadGraph graph, Resource r) throws DatabaseException {
        Layer0 L0 = Layer0.getInstance(graph);
        String name = graph.getPossibleRelatedValue2(r, L0.HasLabel, Bindings.STRING);
        if (name == null || name.isEmpty())
            name = graph.getPossibleRelatedValue(r, L0.HasName, Bindings.STRING);
        if (name == null)
            name = NameUtils.getSafeName(graph, r);
        return name;
    }

}
