/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.flag;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteGraph;
import org.simantics.db.common.primitiverequest.OrderedSet;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.AbstractStringModifier;
import org.simantics.diagram.stubs.DiagramResource;
import org.simantics.layer0.Layer0;
import org.simantics.layer0.utils.binaryPredicates.OrderedSetElementsPredicate;

/**
 * Makes sure that two things apply:
 * <ol>
 * <li>If a flag is joined with another flag, both flags must have the same label property</li>
 * <li>A flag's label is unique within its own diagram</li>
 * </ol> 
 * @author Tuukka Lehtonen
 */
public final class FlagLabelModifier extends AbstractStringModifier {

    private Resource             ownerFlag;
    private Resource             relation;
    @SuppressWarnings("unused")
    private Resource             property;
    private Collection<Resource> diagrams = Collections.emptySet();
    private Set<String>          originalLabels = new HashSet<String>();
    private Set<String>          labelsInUse = Collections.emptySet();

    public FlagLabelModifier(ReadGraph graph, Resource flag, Resource relation, Resource label) throws DatabaseException {
    	super(label);
        this.ownerFlag = flag;
        this.relation = relation;
        this.property = label;
        initialize(graph);
    }
    	
    private void initialize(ReadGraph graph) throws DatabaseException {
        if (ownerFlag != null) {
            diagrams = OrderedSetElementsPredicate.INSTANCE.getSubjects(graph, ownerFlag);
            Resource correspondence = FlagUtil.getPossibleCounterpart(graph, ownerFlag);

            addPossibleLabel(graph, ownerFlag, originalLabels);
            if (correspondence != null)
                addPossibleLabel(graph, correspondence, originalLabels);
        }
        refreshUsedLabels(graph);
    }

    private void addPossibleLabel(ReadGraph graph, Resource flag, Set<String> result) throws DatabaseException {
        String label = graph.getPossibleRelatedValue(flag, Layer0.getInstance(graph).HasLabel, Bindings.STRING);
        if (label != null && !label.isEmpty())
            result.add(label);
    }

    private void refreshUsedLabels(ReadGraph graph) throws DatabaseException {
        Set<String> used = new HashSet<String>();
        DiagramResource DIA = DiagramResource.getInstance(graph);
        for (Resource diagram : diagrams) {
            for (Resource element : graph.syncRequest(new OrderedSet(diagram))) {
                if (graph.isInstanceOf(element, DIA.Flag)) {
                    addPossibleLabel(graph, element, used);
                }
            }
        }
        //System.out.println("used labels:" + used);
        //System.out.println("original labels:" + originalLabels);
        used.removeAll(originalLabels);

        this.labelsInUse = used;
        //System.out.println("final used labels:" + used);
    }

    @Override
    public String isValid(String value) {
        if (diagrams.isEmpty() || ownerFlag == null)
            // No diagram or flag for property, cannot verify, therefore don't care.
            return null;

        if (labelsInUse.contains(value))
            return "Label is already in use on the same diagram";
        return null;
    }

    @Override
    final public void modify(WriteGraph graph, String value) throws DatabaseException {
        DiagramResource DIA = DiagramResource.getInstance(graph);

        graph.claimLiteral(ownerFlag, relation, DIA.FlagLabel, value, Bindings.STRING);
        Resource counterpart = FlagUtil.getPossibleCounterpart(graph, ownerFlag);
        if (counterpart != null)
            graph.claimLiteral(counterpart, relation, DIA.FlagLabel, value, Bindings.STRING);
    }

}
