/*******************************************************************************
 * Copyright (c) 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.diagram.connection.rendering;

import gnu.trove.set.hash.THashSet;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.Path2D;
import java.io.Serializable;
import java.util.List;

import org.simantics.diagram.connection.RouteGraph;
import org.simantics.diagram.connection.RouteLine;
import org.simantics.diagram.connection.RoutePoint;
import org.simantics.diagram.connection.RouteTerminal;
import org.simantics.diagram.connection.rendering.arrows.ILineEndStyle;


public class StyledRouteGraphRenderer implements IRouteGraphRenderer, Serializable {

    private static final BasicStroke GUIDE_LINE_STROKE = new BasicStroke(0.1f);
    private static final Color GUIDE_LINE_COLOR = new Color(255,255,255);
    
	private static final long serialVersionUID = 1564960933064029020L;

	protected final ConnectionStyle style;

	private static class Cache {
		Path2D path = new Path2D.Double();
		THashSet<RoutePoint> branchPoints = new THashSet<>();
	}

	// Caches to avoid creating new objects all the time during rendering
	protected static ThreadLocal<Cache> caches = ThreadLocal.withInitial(() -> new Cache());

	public StyledRouteGraphRenderer(ConnectionStyle style) {
		if (style == null)
			throw new NullPointerException("null style");
		this.style = style;
	}

	public ConnectionStyle getStyle() {
		return style;
	}

	@Override
	public void render(Graphics2D g, RouteGraph rg) {
		Cache cache = caches.get();
		Path2D path = cache.path;
		THashSet<RoutePoint> branchPoints = cache.branchPoints;

		path.reset();
		rg.getPath2D(path);
		style.drawPath(g, path, false);

		branchPoints.clear();
		for(RouteLine line : rg.getLines()) {
		    renderLine(g, line, false);
		    collectBranchPoints(line, branchPoints);
		}
		for(RouteLine line : rg.getTransientLines()) {
			renderLine(g, line, true);
			collectBranchPoints(line, branchPoints);
		}

		/*for(RouteTerminal terminal : rg.getTerminals())
			terminal.render(g);*/
		for(RoutePoint point : branchPoints) {
			style.drawBranchPoint(g, point.getX(), point.getY());
		}
	}
	
	private static void collectBranchPoints(RouteLine line, THashSet<RoutePoint> branchPoints) {
		List<RoutePoint> points = line.getPoints();
		for(int i=1;i<points.size()-1;++i) {
			RoutePoint point = points.get(i);
			branchPoints.add(point);
		}
	}
	
	private void renderLine(Graphics2D g, RouteLine line, boolean isTransient) {
	    if(line.getPoints().size() == 0) {
	        System.err.println("Route line does not contain any points (data = " + line.getData() + ").");
	        return;
	    }
		RoutePoint p1 = line.getBegin();
		RoutePoint p2 = line.getEnd();
		double x1 = p1.getX();
		double y1 = p1.getY();
		double x2 = p2.getX();
		double y2 = p2.getY();
		boolean isHorizontal = line.isHorizontal();
//		double length = isHorizontal ? x2-x1 : y2-y1; // original length before removing the endpoints
		if(isHorizontal) {
			if(p1 instanceof RouteTerminal) {
				RouteTerminal terminal = (RouteTerminal)p1;
				ILineEndStyle style = terminal.getRenderStyle();
				style.render(g, x1, y1, 2);
				x1 += style.getLineEndLength(0);
			}
			if(p2 instanceof RouteTerminal) {
				RouteTerminal terminal = (RouteTerminal)p2;
				ILineEndStyle style = terminal.getRenderStyle();
				style.render(g, x2, y2, 0);
				x2 -= style.getLineEndLength(2);
			}
		}
		else {
			if(p1 instanceof RouteTerminal) {
				RouteTerminal terminal = (RouteTerminal)p1;
				ILineEndStyle style = terminal.getRenderStyle();
				style.render(g, x1, y1, 3);
				y1 += style.getLineEndLength(1);
			}
			if(p2 instanceof RouteTerminal) {
				RouteTerminal terminal = (RouteTerminal)p2;
				ILineEndStyle style = terminal.getRenderStyle();
				style.render(g, x2, y2, 1);
				y2 -= style.getLineEndLength(3);
			}
		}
//		if(length <= style.getDegeneratedLineLength())
//			style.drawDegeneratedLine(g, 0.5*(x1+x2), 0.5*(y1+y2), isHorizontal, isTransient);
//		else
//			style.drawLine(g, x1, y1, x2, y2, isTransient);
	}
	
	@Override
	public void renderGuides(Graphics2D g, RouteGraph rg) {
	    Path2D path = new Path2D.Double();
        for(RouteLine line : rg.getLines()) {
            RoutePoint p1 = line.getBegin();
            RoutePoint p2 = line.getEnd();
            double dx = p2.getX() - p1.getX();
            double dy = p2.getY() - p1.getY();
            double adx = Math.abs(dx);
            double ady = Math.abs(dy);
            if(adx > ady) {
                if(adx > 4)
                    dx = 0.5*dx - Math.signum(dx);
                else
                    dx *= 0.25;
                path.moveTo(p1.getX()+dx, p1.getY());
                path.lineTo(p2.getX()-dx, p2.getY());
            }
            else {
                if(ady > 4)
                    dy = 0.5*dy - Math.signum(dy);
                else
                    dy *= 0.25;
                path.moveTo(p1.getX(), p1.getY()+dy);
                path.lineTo(p2.getX(), p2.getY()-dy);
            }
        }
        g.setStroke(GUIDE_LINE_STROKE);
        g.setColor(GUIDE_LINE_COLOR);
        g.draw(path);
	}

	@Override
	public int hashCode() {
		return style.hashCode();
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		StyledRouteGraphRenderer other = (StyledRouteGraphRenderer) obj;
		return style.equals(other.style);
	}

}
