package fi.vtt.simantics.procore.internal;

import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.exception.ResourceNotFoundException;
import org.simantics.db.exception.RuntimeDatabaseException;
import org.simantics.db.impl.ClusterI;
import org.simantics.db.impl.ResourceImpl;
import org.simantics.db.impl.TransientGraph;
import org.simantics.db.procore.cluster.ClusterImpl;
import org.simantics.db.service.ClusteringSupport;

public class ClusteringSupportImpl implements ClusteringSupport {

    final private SessionImplSocket session;

    ClusteringSupportImpl(SessionImplSocket session) {
        this.session = session;
    }

    @Override
    public long createCluster() {
        long id;
        try {
            id = session.graphSession.newClusterId();
        } catch (DatabaseException e) {
            throw new RuntimeDatabaseException("Failed to get new cluster id.", e);
        }
        session.clusterTable.makeCluster(id, session.writeOnly);
        return id;
    }

    @Override
    public long getCluster(Resource r) {
        int id = session.querySupport.getId(r);
        if(id < 0) // Virtual resource
            return TransientGraph.getVirtualClusterKey(id);
        return session.clusterTable.getClusterIdByResourceKeyNoThrow(id);
    }

    @Override
    public int getNumberOfResources(long clusterId) throws DatabaseException {
        return session.clusterTable.getClusterByClusterId(clusterId).getNumberOfResources(session.clusterTranslator);
    }

    @Override
    public Resource getResourceByKey(int resourceKey)
            throws ResourceNotFoundException {
        return session.getResourceByKey(resourceKey);
    }

    @Override
    public Resource getResourceByIndexAndCluster(int resourceIndex, long clusterId)
            throws DatabaseException, ResourceNotFoundException {
        if (resourceIndex < 1)
            throw new ResourceNotFoundException("Illegal resource index=" + resourceIndex + " cluster=" + clusterId);
        ClusterI cluster = session.getClusterTable().getLoadOrThrow(clusterId);
        int n = cluster.getNumberOfResources(null);
        if (resourceIndex > n)
            throw new ResourceNotFoundException("Illegal resource index=" + resourceIndex + " cluster=" + clusterId
                    + " max index=" + n);
        return session.getResource(resourceIndex, clusterId);
    }
//    @Override
//    public Id getModifiedId(long clusterId) throws DatabaseException {
//        return session.clusterTable.getClusterByClusterId(clusterId).getModifiedId();
//    }

    @Override
    public Resource getClusterSetOfCluster(Resource r) throws DatabaseException {
        if(!r.isPersistent()) return null;
        ClusterImpl cluster = session.clusterTable.getClusterByResourceKey(((ResourceImpl)r).id);
        Long rid = session.clusterSetsSupport.getSet(cluster.getClusterId());
        if(rid == null || rid == 0) return null;
        return session.resourceSerializer.getResource(rid);
    }

    @Override
    public Resource getClusterSetOfCluster(long cluster) throws DatabaseException {
        Long rid = session.clusterSetsSupport.getSet(cluster);
        if(rid == null || rid == 0) return null;
        return session.resourceSerializer.getResource(rid);
    }

    @Override
    public boolean isClusterSet(Resource r) throws DatabaseException {
        return session.containsClusterSet(r);
    }

}
