package org.simantics.db.layer0.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.common.NamedResource;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.utils.CommonDBUtils;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.layer0.adapter.SubgraphExtent.ExtentStatus;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2.SeedSpec;
import org.simantics.db.layer0.util.TransferableGraphConfiguration2.SeedSpec.SeedSpecType;
import org.simantics.layer0.Layer0;

public class TGConfigurer {

	final private ReadGraph graph;
	final Collection<SeedSpec> roots = new ArrayList<>();
	final Map<Resource, ExtentStatus> preStatus = new HashMap<>();
	final boolean ignoreVirtualResources;
	final boolean validate;
	
	public TGConfigurer(ReadGraph graph, boolean ignoreVirtualResources, boolean validate) {
		this.graph = graph;
		this.ignoreVirtualResources = ignoreVirtualResources;
		this.validate = validate;
	}
	
	public TGConfigurer(ReadGraph graph, boolean ignoreVirtualResources) {
		this(graph, ignoreVirtualResources, true);
	}

	public TGConfigurer exclusions(Collection<Resource> exclusions) {
		for(Resource r : exclusions) {
			preStatus.put(r, ExtentStatus.EXCLUDED);
		}
		return this;
	}

	public TGConfigurer roots(Collection<SeedSpec> rs) throws DatabaseException {
		Layer0 L0 = Layer0.getInstance(graph);
		for(SeedSpec root : rs) {
			Resource nameResource = graph.getPossibleObject(root.resource, L0.HasName);
			if(nameResource != null) preStatus.put(nameResource, ExtentStatus.EXCLUDED);
			roots.add(new SeedSpec(root.resource, root.name, root.specType, root.type));
			preStatus.put(root.resource, ExtentStatus.INTERNAL);
		}
		return this;
	}

	public TGConfigurer roots2(Collection<Resource> rs) throws DatabaseException {
		Layer0 L0 = Layer0.getInstance(graph);
		for(Resource root : rs) {
			Resource nameResource = graph.getPossibleObject(root, L0.HasName);
			String name = "<no name>";
			if(nameResource != null) {
				name = graph.getPossibleValue(nameResource, Bindings.STRING);
				if(name == null) name = "<no name>";
				preStatus.put(nameResource, ExtentStatus.EXCLUDED);
			}
			roots.add(new SeedSpec(root, name, SeedSpecType.ROOT));
			preStatus.put(root, ExtentStatus.INTERNAL);
		}
		return this;
	}
	
	public TGConfigurer roots3(Collection<NamedResource> rs) throws DatabaseException {
		Layer0 L0 = Layer0.getInstance(graph);
		for(NamedResource root : rs) {
			Resource nameResource = graph.getPossibleObject(root.getResource(), L0.HasName);
			roots.add(new SeedSpec(root.getResource(), root.getName(), SeedSpecType.ROOT));
			preStatus.put(root.getResource(), ExtentStatus.INTERNAL);
			preStatus.put(nameResource, ExtentStatus.EXCLUDED);
		}
		return this;
	}

	public TransferableGraphConfiguration2 create() throws DatabaseException {

		// Final preparation

		// -Exclude root library if not explicitly included
		if (!roots.contains(graph.getRootLibrary()))
			preStatus.put(graph.getRootLibrary(), ExtentStatus.EXCLUDED);

		// -Exclude owners if not explicitly defined
		Layer0 L0 = Layer0.getInstance(graph);
		ArrayList<Resource> rootList = new ArrayList<>();
		for (SeedSpec root : roots) {
			if(!SeedSpecType.SPECIAL_ROOT.equals(root.specType)) {
				rootList.add(root.resource);
				for (Resource owner : graph.getObjects(root.resource, L0.IsOwnedBy)) {
					ExtentStatus ownerStatus = preStatus.get(owner);
					if(ownerStatus == null)
						preStatus.put(owner, ExtentStatus.EXCLUDED);
				}
			} else {
				preStatus.put(root.resource, ExtentStatus.EXTERNAL);
			}
		}
		
		Resource owner = CommonDBUtils.getNearestOwner(graph, rootList);
		if(owner != null) {
			Resource indexRoot = graph.syncRequest(new PossibleIndexRoot(owner));
			return new TransferableGraphConfiguration2(indexRoot, roots, preStatus, ignoreVirtualResources, validate);
		} else {
			return new TransferableGraphConfiguration2(null, roots, preStatus, ignoreVirtualResources, validate);
		}
		
	}
	
}
