/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.layer0.adapter.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.simantics.databoard.Bindings;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.WriteOnlyGraph;
import org.simantics.db.common.request.FreshEscapedName;
import org.simantics.db.common.request.PossibleIndexRoot;
import org.simantics.db.common.utils.Versions;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.service.SerialisationSupport;
import org.simantics.graph.db.AbstractImportAdvisor2;
import org.simantics.graph.db.ImportAdvisors;
import org.simantics.graph.db.TransferableGraphImporter;
import org.simantics.graph.representation.Root;
import org.simantics.layer0.Layer0;

public class DefaultPasteImportAdvisor extends AbstractImportAdvisor2 {

	protected Resource library;
	protected Resource temp;
	protected Resource model;
	protected String singleType = null;
	protected long[] ids;
	protected Map<String, String> nameMappings = new HashMap<String, String>();
	
	final protected Map<String, Object> context;

	public DefaultPasteImportAdvisor(ReadGraph graph, Resource library) throws DatabaseException {
		this(library, graph.syncRequest(new PossibleIndexRoot(library)), Collections.<String,Object>emptyMap());
	}

	public DefaultPasteImportAdvisor(Resource library) {
		this(library, library, Collections.<String,Object>emptyMap());
	}

	public DefaultPasteImportAdvisor(Resource library, Map<String, Object> context) {
		this(library, library, context);
	}

	public DefaultPasteImportAdvisor(Resource library, Resource model, Map<String, Object> context) {
		this.library = library;
		this.model = model;
		this.context = context;
	}

	@Override
	public void redirect(Resource temp) {
		this.temp = temp;
	}
	
	private Resource getRedirectTarget() {
		if(temp != null) return temp;
		else return library;
	}
	
	public Resource getTarget() {
		return library;
	}
	
	public void analyzeType(ReadGraph graph, Root root) throws DatabaseException {
	}
	
	@Override
	public Resource analyzeRoot(ReadGraph graph, Root root) throws DatabaseException {
		
		if("%model".equals(root.name)) {
			return model;
		}

		analyzeType(graph, root);

		String type = root.type;
		if(singleType != null) {
			if(!type.equals(singleType)) throw new DatabaseException("Paste of a set of different types of objects is not supported.");
		} else {
			singleType = type;
		}
		
		if(library != null) {
			String newName = newName(graph, library, root.name);
			nameMappings.put(root.name, newName);
		}
		
		return null;

	}
	
	public String newName(ReadGraph graph, Resource library, String name) throws DatabaseException {
		
		Map<String,String> renameMap = (Map<String,String>)context.get(ImportAdvisors.RENAME_MAP);
		if(renameMap != null) {
			String renamed = renameMap.get(name);
			if(renamed != null) return renamed;
		}
		

		String version = Versions.getVersion(name);
		if(version != null) {
			String baseName = Versions.getBaseName(name);
			String newBaseName = graph.syncRequest(new FreshEscapedName(library, Layer0.getInstance(graph).ConsistsOf, baseName));
			return Versions.make(newBaseName, Versions.getBaseVersion(version));
		} else {
			return graph.syncRequest(new FreshEscapedName(library, Layer0.getInstance(graph).ConsistsOf, name)); 
		}
		
	}

	@Override
	public Resource createRoot(WriteOnlyGraph graph, Root root) throws DatabaseException {
		return createRoot(graph, root, null);
	}
	
	public String getName(Root root) {
		String name = root.name;
		String newName = nameMappings.get(name);
		if(newName != null) name = newName;
		return name;
	}
	
	@Override
	public Resource createRoot(WriteOnlyGraph graph, Root root, Resource resource) throws DatabaseException {
		
		Layer0 l0 = graph.getService(Layer0.class);

		if(resource == null) resource = graph.newResource();
		if(getRedirectTarget() != null) {
			graph.claim(getRedirectTarget(), l0.ConsistsOf, l0.PartOf, resource);
		}
		String newName = getName(root);
		graph.addLiteral(resource, l0.HasName, l0.NameOf, l0.String, newName, Bindings.STRING);
		
		addRootInfo(root, newName, resource);
		
		return resource;
		
	}

    @Override
    public void beforeWrite(WriteOnlyGraph graph, TransferableGraphImporter process)
            throws DatabaseException {
    }

    @Override
    public void afterWrite(WriteOnlyGraph graph, TransferableGraphImporter process)
            throws DatabaseException {
    	ids = process.getResourceIds(graph.getService(SerialisationSupport.class));
    }
    
    @Override
    public boolean allowImmutableModifications() {
        return false;
    }
	
    @Override
    public Resource createChild(WriteOnlyGraph graph, TransferableGraphImporter process, Resource parent,
            String name) throws DatabaseException {
        return process.createChild(graph, parent, null, name);
    }
    
    @Override
    public Resource createChild(WriteOnlyGraph graph, TransferableGraphImporter process, Resource parent, Resource child,
            String name) throws DatabaseException {
        return process.createChild(graph, parent, child, name);
    }
    
    public long[] getResourceIds() {
    	return ids;
    }
    
}
