/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.request.Read;
import org.simantics.db.request.ReadExt;
import org.simantics.db.request.RequestFlags;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ReadEntry<T> extends CacheEntryBase<AsyncProcedure<T>> implements AsyncProcedure<T> {

    private static final Logger LOGGER = LoggerFactory.getLogger(ReadEntry.class);

    protected Read<T> request;

    public ReadEntry(Read<T> request) {
        this.request = request;
    }

    @Override
    int makeHash() {
        return request.hashCode();
    }

    @Override
    public Object getOriginalRequest() {
        return request;
    }

    @Override
    public void discard() {
        super.discard();
        setResult(null);
    }

    @Override
    final public Query getQuery() {

        return new Query() {

            @Override
            public void recompute(ReadGraphImpl graph) {

                try {

                    T result = request.perform(graph);
                    setResult(result);
                    setReady();

                } catch (Throwable t) {

                    except(t);

                }

            }

            @Override
            public void removeEntry(QueryProcessor processor) {
                processor.cache.remove(ReadEntry.this);
            }

            @Override
            public int type() {
                if (request instanceof ReadExt) {
                    return ((ReadExt) request).getType();
                } else {
                    return RequestFlags.INVALIDATE;
                }
            }

            @Override
            public String toString() {
                if (request == null)
                    return "DISCARDED";
                else
                    return request.toString() + statusOrException;
            }

        };

    }

    public static <T> T computeForEach(ReadGraphImpl graph, Read<T> request, ReadEntry<T> entry,
            AsyncProcedure<T> procedure_) throws DatabaseException {

        AsyncProcedure<T> procedure = entry != null ? entry : procedure_;

        ReadGraphImpl queryGraph = graph.withParent(entry);
        queryGraph.asyncBarrier.inc();

        ReadGraphImpl executeGraph = graph.withParent(graph.parent);
        executeGraph.asyncBarrier.inc();
        
        try {

            // This throws
            T result = request.perform(queryGraph);

            if(procedure != null) procedure.execute(executeGraph, result);
            return (T)result;

        } catch (DatabaseException e) {

            if(procedure != null) procedure.exception(executeGraph, e);
            throw e;

        } catch (Throwable t) {

            DatabaseException dbe = new DatabaseException(t);
            if(procedure != null) procedure.exception(executeGraph, dbe);
            throw dbe;

        } finally {

            queryGraph.asyncBarrier.dec();

            try {
            
                if (entry != null) {
                    // This also throws so must dec barrier finally
                    entry.performFromCache(executeGraph, procedure_);
                }
            
            } finally {
 
                executeGraph.asyncBarrier.dec();
                executeGraph.asyncBarrier.waitBarrier(procedure, executeGraph);

            }
                
 
        }

    }

    public Object performFromCache(ReadGraphImpl graph, AsyncProcedure<T> procedure) throws DatabaseException {

        AsyncProcedure<T> proc = (AsyncProcedure<T>) procedure;

        if (isExcepted()) {
                if(proc != null) {
                    try {
                        proc.exception(graph, (Throwable) getResult());
                    } catch (Throwable t) {
                        LOGGER.error("performFromCache proc.exception failed", t);
                    }
                }
                Throwable t = (Throwable) getResult();
                if(t instanceof DatabaseException) {
                    throw (DatabaseException)t;
                } else {
                    throw new DatabaseException(t);
                }
        } else {
                if(proc != null) {
                    try {
                        proc.execute(graph, (T) getResult());
                    } catch (Throwable t) {
                        LOGGER.error("performFromCache proc.execute failed", t);
                    }
                }
                return (T)getResult();
        }

    }

    @Override
    public String toString() {
        if (request == null)
            return "DISCARDED";
        else
            return request.toString() + " - " + statusOrException;
    }

    public Object get(ReadGraphImpl graph, AsyncProcedure<T> procedure) throws DatabaseException {
        if (procedure != null)
            performFromCache(graph, procedure);
        checkAndThrow();
        return getResult();
    }

    @Override
    boolean isImmutable(ReadGraphImpl graph) throws DatabaseException {
        if (request instanceof ReadExt) {
            return ((ReadExt) request).isImmutable(graph);
        }
        return false;
    }

    @Override
    public void execute(AsyncReadGraph graph, T result) {
        setResult(result);
        setReady();
    }

    @Override
    public void exception(AsyncReadGraph graph, Throwable throwable) {
        except(throwable);
    }

}
