package org.simantics.db.impl.query;

import org.simantics.db.ObjectResourceIdMap;
import org.simantics.db.RelationInfo;
import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.impl.procedure.InternalProcedure;
import org.simantics.db.impl.query.QueryProcessor.SessionTask;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.procedure.AsyncProcedure;
import org.simantics.db.procedure.ListenerBase;
import org.simantics.db.procedure.SyncMultiProcedure;
import org.simantics.db.request.AsyncMultiRead;
import org.simantics.db.request.AsyncRead;
import org.simantics.db.request.ExternalRead;
import org.simantics.db.request.MultiRead;
import org.simantics.db.request.Read;

public class QueryCache extends QueryCacheBase {

    public QueryCache(QuerySupport querySupport, int threads) {
        super(querySupport, threads);
    }

    Objects getOrCreateObjects(ReadGraphImpl graph, int r1, int r2) throws DatabaseException {
        Objects existing = null;
        synchronized(objectsMap) {
            existing = (Objects)objectsMap.get(r1,r2);
            if(existing == null) {
                existing = new Objects(r1,r2);
                existing.clearResult(querySupport);
                existing.setPending();
                objectsMap.put(keyR2(r1,r2), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(Objects entry) {
        synchronized(objectsMap) {
            objectsMap.remove(entry.id);
        }
    }
    
    public static void runnerObjects(ReadGraphImpl graph, int r1, int r2, CacheEntry parent, ListenerBase listener, final IntProcedure procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r1,r2)) {
            Objects.computeForEach(graph, r1,r2, null, procedure);
            return;
        }
        Objects entry = (Objects)cache.getOrCreateObjects(graph, r1,r2);
        IntProcedure procedure_ = procedure != null ? procedure : emptyProcedureObjects;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          Objects.computeForEach(graph, r1,r2, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    Statements getOrCreateStatements(ReadGraphImpl graph, int r1, int r2) throws DatabaseException {
        Statements existing = null;
        synchronized(statementsMap) {
            existing = (Statements)statementsMap.get(r1,r2);
            if(existing == null) {
                existing = new Statements(r1,r2);
                existing.clearResult(querySupport);
                existing.setPending();
                statementsMap.put(keyR2(r1,r2), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(Statements entry) {
        synchronized(statementsMap) {
            statementsMap.remove(entry.id);
        }
    }
    
    public static void runnerStatements(ReadGraphImpl graph, int r1, int r2, CacheEntry parent, ListenerBase listener, final TripleIntProcedure procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r1,r2)) {
            Statements.computeForEach(graph, r1,r2, null, procedure);
            return;
        }
        Statements entry = (Statements)cache.getOrCreateStatements(graph, r1,r2);
        TripleIntProcedure procedure_ = procedure != null ? procedure : emptyProcedureStatements;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          Statements.computeForEach(graph, r1,r2, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    DirectObjects getOrCreateDirectObjects(ReadGraphImpl graph, int r1, int r2) throws DatabaseException {
        DirectObjects existing = null;
        synchronized(directObjectsMap) {
            existing = (DirectObjects)directObjectsMap.get(r1,r2);
            if(existing == null) {
                existing = new DirectObjects(r1,r2);
                existing.clearResult(querySupport);
                existing.setPending();
                directObjectsMap.put(keyR2(r1,r2), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(DirectObjects entry) {
        synchronized(directObjectsMap) {
            directObjectsMap.remove(entry.id);
        }
    }
    
    public static void runnerDirectObjects(ReadGraphImpl graph, int r1, int r2, CacheEntry parent, ListenerBase listener, final IntProcedure procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r1,r2)) {
            DirectObjects.computeForEach(graph, r1,r2, null, procedure);
            return;
        }
        DirectObjects entry = (DirectObjects)cache.getOrCreateDirectObjects(graph, r1,r2);
        IntProcedure procedure_ = procedure != null ? procedure : emptyProcedureDirectObjects;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          DirectObjects.computeForEach(graph, r1,r2, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    RelationInfoQuery getOrCreateRelationInfoQuery(ReadGraphImpl graph, int r) throws DatabaseException {
        RelationInfoQuery existing = null;
        synchronized(relationInfoQueryMap) {
            existing = (RelationInfoQuery)relationInfoQueryMap.get(r);
            if(existing == null) {
                existing = new RelationInfoQuery(r);
                existing.clearResult(querySupport);
                existing.setPending();
                relationInfoQueryMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(RelationInfoQuery entry) {
        synchronized(relationInfoQueryMap) {
            relationInfoQueryMap.remove(entry.id);
        }
    }
    
    public static void runnerRelationInfoQuery(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final InternalProcedure<RelationInfo> procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            RelationInfoQuery.computeForEach(graph, r, null, procedure);
            return;
        }
        RelationInfoQuery entry = (RelationInfoQuery)cache.getOrCreateRelationInfoQuery(graph, r);
        InternalProcedure<RelationInfo> procedure_ = procedure != null ? procedure : emptyProcedureRelationInfoQuery;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          RelationInfoQuery.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    URIToResource getOrCreateURIToResource(ReadGraphImpl graph, String id) throws DatabaseException {
        URIToResource existing = null;
        synchronized(uRIToResourceMap) {
            existing = (URIToResource)uRIToResourceMap.get(id);
            if(existing == null) {
                existing = new URIToResource(id);
                existing.clearResult(querySupport);
                existing.setPending();
                uRIToResourceMap.put(keyID(id), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(URIToResource entry) {
        synchronized(uRIToResourceMap) {
            uRIToResourceMap.remove(entry.id);
        }
    }
    
    public static void runnerURIToResource(ReadGraphImpl graph, String id, CacheEntry parent, ListenerBase listener, final InternalProcedure<Integer> procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, id)) {
            URIToResource.computeForEach(graph, id, null, procedure);
            return;
        }
        URIToResource entry = (URIToResource)cache.getOrCreateURIToResource(graph, id);
        InternalProcedure<Integer> procedure_ = procedure != null ? procedure : emptyProcedureURIToResource;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          URIToResource.computeForEach(graph, id, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    ValueQuery getOrCreateValueQuery(ReadGraphImpl graph, int r) throws DatabaseException {
        ValueQuery existing = null;
        synchronized(valueQueryMap) {
            existing = (ValueQuery)valueQueryMap.get(r);
            if(existing == null) {
                existing = new ValueQuery(r);
                existing.clearResult(querySupport);
                existing.setPending();
                valueQueryMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(ValueQuery entry) {
        synchronized(valueQueryMap) {
            valueQueryMap.remove(entry.id);
        }
    }
    
    public static void runnerValueQuery(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final InternalProcedure<byte[]> procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            ValueQuery.computeForEach(graph, r, null, procedure);
            return;
        }
        ValueQuery entry = (ValueQuery)cache.getOrCreateValueQuery(graph, r);
        InternalProcedure<byte[]> procedure_ = procedure != null ? procedure : emptyProcedureValueQuery;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          ValueQuery.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    OrderedSet getOrCreateOrderedSet(ReadGraphImpl graph, int r) throws DatabaseException {
        OrderedSet existing = null;
        synchronized(orderedSetMap) {
            existing = (OrderedSet)orderedSetMap.get(r);
            if(existing == null) {
                existing = new OrderedSet(r);
                existing.clearResult(querySupport);
                existing.setPending();
                orderedSetMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(OrderedSet entry) {
        synchronized(orderedSetMap) {
            orderedSetMap.remove(entry.id);
        }
    }
    
    public static void runnerOrderedSet(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final IntProcedure procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            OrderedSet.computeForEach(graph, r, null, procedure);
            return;
        }
        OrderedSet entry = (OrderedSet)cache.getOrCreateOrderedSet(graph, r);
        IntProcedure procedure_ = procedure != null ? procedure : emptyProcedureOrderedSet;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          OrderedSet.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    PrincipalTypes getOrCreatePrincipalTypes(ReadGraphImpl graph, int r) throws DatabaseException {
        PrincipalTypes existing = null;
        synchronized(principalTypesMap) {
            existing = (PrincipalTypes)principalTypesMap.get(r);
            if(existing == null) {
                existing = new PrincipalTypes(r);
                existing.clearResult(querySupport);
                existing.setPending();
                principalTypesMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(PrincipalTypes entry) {
        synchronized(principalTypesMap) {
            principalTypesMap.remove(entry.id);
        }
    }
    
    public static void runnerPrincipalTypes(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final IntProcedure procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            PrincipalTypes.computeForEach(graph, r, null, procedure);
            return;
        }
        PrincipalTypes entry = (PrincipalTypes)cache.getOrCreatePrincipalTypes(graph, r);
        IntProcedure procedure_ = procedure != null ? procedure : emptyProcedurePrincipalTypes;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          PrincipalTypes.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    DirectPredicates getOrCreateDirectPredicates(ReadGraphImpl graph, int r) throws DatabaseException {
        DirectPredicates existing = null;
        synchronized(directPredicatesMap) {
            existing = (DirectPredicates)directPredicatesMap.get(r);
            if(existing == null) {
                existing = new DirectPredicates(r);
                existing.clearResult(querySupport);
                existing.setPending();
                directPredicatesMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(DirectPredicates entry) {
        synchronized(directPredicatesMap) {
            directPredicatesMap.remove(entry.id);
        }
    }
    
    public static void runnerDirectPredicates(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final InternalProcedure<IntSet> procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            DirectPredicates.computeForEach(graph, r, null, procedure);
            return;
        }
        DirectPredicates entry = (DirectPredicates)cache.getOrCreateDirectPredicates(graph, r);
        InternalProcedure<IntSet> procedure_ = procedure != null ? procedure : emptyProcedureDirectPredicates;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          DirectPredicates.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    Predicates getOrCreatePredicates(ReadGraphImpl graph, int r) throws DatabaseException {
        Predicates existing = null;
        synchronized(predicatesMap) {
            existing = (Predicates)predicatesMap.get(r);
            if(existing == null) {
                existing = new Predicates(r);
                existing.clearResult(querySupport);
                existing.setPending();
                predicatesMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(Predicates entry) {
        synchronized(predicatesMap) {
            predicatesMap.remove(entry.id);
        }
    }
    
    public static void runnerPredicates(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final InternalProcedure<IntSet> procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            Predicates.computeForEach(graph, r, null, procedure);
            return;
        }
        Predicates entry = (Predicates)cache.getOrCreatePredicates(graph, r);
        InternalProcedure<IntSet> procedure_ = procedure != null ? procedure : emptyProcedurePredicates;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          Predicates.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    ReadEntry getOrCreateReadEntry(ReadGraphImpl graph, Read<?> r, boolean needsToBlock) throws DatabaseException {
        ReadEntry existing = null;
        synchronized(readEntryMap) {
            existing = (ReadEntry)readEntryMap.get(r);
            if(existing == null) {
                existing = new ReadEntry(r);
                existing.clearResult(querySupport);
                existing.setPending();
                readEntryMap.put(id(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) {
            if(needsToBlock) waitPending(graph, existing);
            else return null;
        }
        return existing;
    }
    
    void remove(ReadEntry entry) {
        synchronized(readEntryMap) {
            readEntryMap.remove(entry.request);
        }
    }
    
    public static Object runnerReadEntry(ReadGraphImpl graph, Read<?> r, CacheEntry parent, ListenerBase listener, final AsyncProcedure procedure, final boolean needsToBlock) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            return ReadEntry.computeForEach(graph, r, null, procedure);
        }
        ReadEntry entry = (ReadEntry)cache.getOrCreateReadEntry(graph, r, needsToBlock);
        if(entry == null) {
            graph.processor.schedule(new SessionTask(graph) {
                @Override
                public void run0(int thread) {
                    try {
                        runnerReadEntry(graph, r, parent, listener, procedure, needsToBlock);
                    } catch (DatabaseException e) {
                        Logger.defaultLogError(e);
                    }
                }
            });
            return null;
        }
        AsyncProcedure procedure_ = procedure != null ? procedure : emptyProcedureReadEntry;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) return entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          Object result = ReadEntry.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
          return result;
        }
    }
    
    AsyncReadEntry getOrCreateAsyncReadEntry(ReadGraphImpl graph, AsyncRead<?> r, boolean needsToBlock) throws DatabaseException {
        AsyncReadEntry existing = null;
        synchronized(asyncReadEntryMap) {
            existing = (AsyncReadEntry)asyncReadEntryMap.get(r);
            if(existing == null) {
                existing = new AsyncReadEntry(r);
                existing.clearResult(querySupport);
                existing.setPending();
                asyncReadEntryMap.put(id(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) {
            if(needsToBlock) waitPending(graph, existing);
            else return null;
        }
        return existing;
    }
    
    void remove(AsyncReadEntry entry) {
        synchronized(asyncReadEntryMap) {
            asyncReadEntryMap.remove(entry.request);
        }
    }
    
    public static Object runnerAsyncReadEntry(ReadGraphImpl graph, AsyncRead<?> r, CacheEntry parent, ListenerBase listener, final AsyncProcedure procedure, final boolean needsToBlock) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            return AsyncReadEntry.computeForEach(graph, r, null, procedure, needsToBlock);
        }
        AsyncReadEntry entry = (AsyncReadEntry)cache.getOrCreateAsyncReadEntry(graph, r, needsToBlock);
        if(entry == null) {
            graph.processor.schedule(new SessionTask(graph) {
                @Override
                public void run0(int thread) {
                    try {
                        runnerAsyncReadEntry(graph, r, parent, listener, procedure, needsToBlock);
                    } catch (DatabaseException e) {
                        Logger.defaultLogError(e);
                    }
                }
            });
            return null;
        }
        AsyncProcedure procedure_ = procedure != null ? procedure : emptyProcedureAsyncReadEntry;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) return entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          Object result = AsyncReadEntry.computeForEach(graph, r, entry, procedure_, needsToBlock);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
          return result;
        }
    }
    
    Types getOrCreateTypes(ReadGraphImpl graph, int r) throws DatabaseException {
        Types existing = null;
        synchronized(typesMap) {
            existing = (Types)typesMap.get(r);
            if(existing == null) {
                existing = new Types(r);
                existing.clearResult(querySupport);
                existing.setPending();
                typesMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(Types entry) {
        synchronized(typesMap) {
            typesMap.remove(entry.id);
        }
    }
    
    public static void runnerTypes(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final InternalProcedure<IntSet> procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            Types.computeForEach(graph, r, null, procedure);
            return;
        }
        Types entry = (Types)cache.getOrCreateTypes(graph, r);
        InternalProcedure<IntSet> procedure_ = procedure != null ? procedure : emptyProcedureTypes;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          Types.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    ChildMap getOrCreateChildMap(ReadGraphImpl graph, int r) throws DatabaseException {
        ChildMap existing = null;
        synchronized(childMapMap) {
            existing = (ChildMap)childMapMap.get(r);
            if(existing == null) {
                existing = new ChildMap(r);
                existing.clearResult(querySupport);
                existing.setPending();
                childMapMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(ChildMap entry) {
        synchronized(childMapMap) {
            childMapMap.remove(entry.id);
        }
    }
    
    public static void runnerChildMap(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final InternalProcedure<ObjectResourceIdMap<String>> procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            ChildMap.computeForEach(graph, r, null, procedure);
            return;
        }
        ChildMap entry = (ChildMap)cache.getOrCreateChildMap(graph, r);
        InternalProcedure<ObjectResourceIdMap<String>> procedure_ = procedure != null ? procedure : emptyProcedureChildMap;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          ChildMap.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    TypeHierarchy getOrCreateTypeHierarchy(ReadGraphImpl graph, int r) throws DatabaseException {
        TypeHierarchy existing = null;
        synchronized(typeHierarchyMap) {
            existing = (TypeHierarchy)typeHierarchyMap.get(r);
            if(existing == null) {
                existing = new TypeHierarchy(r);
                existing.clearResult(querySupport);
                existing.setPending();
                typeHierarchyMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(TypeHierarchy entry) {
        synchronized(typeHierarchyMap) {
            typeHierarchyMap.remove(entry.id);
        }
    }
    
    public static void runnerTypeHierarchy(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final InternalProcedure<IntSet> procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            TypeHierarchy.computeForEach(graph, r, null, procedure);
            return;
        }
        TypeHierarchy entry = (TypeHierarchy)cache.getOrCreateTypeHierarchy(graph, r);
        InternalProcedure<IntSet> procedure_ = procedure != null ? procedure : emptyProcedureTypeHierarchy;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          TypeHierarchy.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    SuperTypes getOrCreateSuperTypes(ReadGraphImpl graph, int r) throws DatabaseException {
        SuperTypes existing = null;
        synchronized(superTypesMap) {
            existing = (SuperTypes)superTypesMap.get(r);
            if(existing == null) {
                existing = new SuperTypes(r);
                existing.clearResult(querySupport);
                existing.setPending();
                superTypesMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(SuperTypes entry) {
        synchronized(superTypesMap) {
            superTypesMap.remove(entry.id);
        }
    }
    
    public static void runnerSuperTypes(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final InternalProcedure<IntSet> procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            SuperTypes.computeForEach(graph, r, null, procedure);
            return;
        }
        SuperTypes entry = (SuperTypes)cache.getOrCreateSuperTypes(graph, r);
        InternalProcedure<IntSet> procedure_ = procedure != null ? procedure : emptyProcedureSuperTypes;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          SuperTypes.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    SuperRelations getOrCreateSuperRelations(ReadGraphImpl graph, int r) throws DatabaseException {
        SuperRelations existing = null;
        synchronized(superRelationsMap) {
            existing = (SuperRelations)superRelationsMap.get(r);
            if(existing == null) {
                existing = new SuperRelations(r);
                existing.clearResult(querySupport);
                existing.setPending();
                superRelationsMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(SuperRelations entry) {
        synchronized(superRelationsMap) {
            superRelationsMap.remove(entry.id);
        }
    }
    
    public static void runnerSuperRelations(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final InternalProcedure<IntSet> procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        if(parent == null && listener == null && !cache.shouldCache(graph.processor, r)) {
            SuperRelations.computeForEach(graph, r, null, procedure);
            return;
        }
        SuperRelations entry = (SuperRelations)cache.getOrCreateSuperRelations(graph, r);
        InternalProcedure<IntSet> procedure_ = procedure != null ? procedure : emptyProcedureSuperRelations;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          SuperRelations.computeForEach(graph, r, entry, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    AssertedPredicates getOrCreateAssertedPredicates(ReadGraphImpl graph, int r) throws DatabaseException {
        AssertedPredicates existing = null;
        synchronized(assertedPredicatesMap) {
            existing = (AssertedPredicates)assertedPredicatesMap.get(r);
            if(existing == null) {
                existing = new AssertedPredicates(r);
                existing.clearResult(querySupport);
                existing.setPending();
                assertedPredicatesMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(AssertedPredicates entry) {
        synchronized(assertedPredicatesMap) {
            assertedPredicatesMap.remove(entry.id);
        }
    }
    
    public static void runnerAssertedPredicates(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final IntProcedure procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        AssertedPredicates entry = (AssertedPredicates)cache.getOrCreateAssertedPredicates(graph, r);
        IntProcedure procedure_ = procedure != null ? procedure : emptyProcedureAssertedPredicates;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          entry.compute(graph, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    AssertedStatements getOrCreateAssertedStatements(ReadGraphImpl graph, int r1, int r2) throws DatabaseException {
        AssertedStatements existing = null;
        synchronized(assertedStatementsMap) {
            existing = (AssertedStatements)assertedStatementsMap.get(r1,r2);
            if(existing == null) {
                existing = new AssertedStatements(r1,r2);
                existing.clearResult(querySupport);
                existing.setPending();
                assertedStatementsMap.put(keyR2(r1,r2), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(AssertedStatements entry) {
        synchronized(assertedStatementsMap) {
            assertedStatementsMap.remove(entry.id);
        }
    }
    
    public static void runnerAssertedStatements(ReadGraphImpl graph, int r1, int r2, CacheEntry parent, ListenerBase listener, final TripleIntProcedure procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        AssertedStatements entry = (AssertedStatements)cache.getOrCreateAssertedStatements(graph, r1,r2);
        TripleIntProcedure procedure_ = procedure != null ? procedure : emptyProcedureAssertedStatements;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          entry.compute(graph, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    DirectSuperRelations getOrCreateDirectSuperRelations(ReadGraphImpl graph, int r) throws DatabaseException {
        DirectSuperRelations existing = null;
        synchronized(directSuperRelationsMap) {
            existing = (DirectSuperRelations)directSuperRelationsMap.get(r);
            if(existing == null) {
                existing = new DirectSuperRelations(r);
                existing.clearResult(querySupport);
                existing.setPending();
                directSuperRelationsMap.put(keyR(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(DirectSuperRelations entry) {
        synchronized(directSuperRelationsMap) {
            directSuperRelationsMap.remove(entry.id);
        }
    }
    
    public static void runnerDirectSuperRelations(ReadGraphImpl graph, int r, CacheEntry parent, ListenerBase listener, final IntProcedure procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        DirectSuperRelations entry = (DirectSuperRelations)cache.getOrCreateDirectSuperRelations(graph, r);
        IntProcedure procedure_ = procedure != null ? procedure : emptyProcedureDirectSuperRelations;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          entry.compute(graph, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    MultiReadEntry getOrCreateMultiReadEntry(ReadGraphImpl graph, MultiRead<?> r) throws DatabaseException {
        MultiReadEntry existing = null;
        synchronized(multiReadEntryMap) {
            existing = (MultiReadEntry)multiReadEntryMap.get(r);
            if(existing == null) {
                existing = new MultiReadEntry(r);
                existing.clearResult(querySupport);
                existing.setPending();
                multiReadEntryMap.put(id(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(MultiReadEntry entry) {
        synchronized(multiReadEntryMap) {
            multiReadEntryMap.remove(entry.request);
        }
    }
    
    public static void runnerMultiReadEntry(ReadGraphImpl graph, MultiRead<?> r, CacheEntry parent, ListenerBase listener, final SyncMultiProcedure procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        MultiReadEntry entry = (MultiReadEntry)cache.getOrCreateMultiReadEntry(graph, r);
        SyncMultiProcedure procedure_ = procedure != null ? procedure : emptyProcedureMultiReadEntry;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          entry.compute(graph, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    AsyncMultiReadEntry getOrCreateAsyncMultiReadEntry(ReadGraphImpl graph, AsyncMultiRead<?> r) throws DatabaseException {
        AsyncMultiReadEntry existing = null;
        synchronized(asyncMultiReadEntryMap) {
            existing = (AsyncMultiReadEntry)asyncMultiReadEntryMap.get(r);
            if(existing == null) {
                existing = new AsyncMultiReadEntry(r);
                existing.clearResult(querySupport);
                existing.setPending();
                asyncMultiReadEntryMap.put(id(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(AsyncMultiReadEntry entry) {
        synchronized(asyncMultiReadEntryMap) {
            asyncMultiReadEntryMap.remove(entry.request);
        }
    }
    
    public static void runnerAsyncMultiReadEntry(ReadGraphImpl graph, AsyncMultiRead<?> r, CacheEntry parent, ListenerBase listener, final AsyncMultiProcedure procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        AsyncMultiReadEntry entry = (AsyncMultiReadEntry)cache.getOrCreateAsyncMultiReadEntry(graph, r);
        AsyncMultiProcedure procedure_ = procedure != null ? procedure : emptyProcedureAsyncMultiReadEntry;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          entry.compute(graph, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
    ExternalReadEntry getOrCreateExternalReadEntry(ReadGraphImpl graph, ExternalRead<?> r) throws DatabaseException {
        ExternalReadEntry existing = null;
        synchronized(externalReadEntryMap) {
            existing = (ExternalReadEntry)externalReadEntryMap.get(r);
            if(existing == null) {
                existing = new ExternalReadEntry(r, graph);
                existing.clearResult(querySupport);
                existing.setPending();
                externalReadEntryMap.put(id(r), existing);
                size++;
                return existing;
            }
            if(existing.requiresComputation()) {
                existing.setPending();
                return existing;
            }
        }
        if(existing.isPending()) waitPending(graph, existing);
        return existing;
    }
    
    void remove(ExternalReadEntry entry) {
        synchronized(externalReadEntryMap) {
            externalReadEntryMap.remove(entry.request);
        }
    }
    
    public static void runnerExternalReadEntry(ReadGraphImpl graph, ExternalRead<?> r, CacheEntry parent, ListenerBase listener, final AsyncProcedure procedure) throws DatabaseException {
        QueryCache cache  = graph.processor.cache;
        ExternalReadEntry entry = (ExternalReadEntry)cache.getOrCreateExternalReadEntry(graph, r);
        AsyncProcedure procedure_ = procedure != null ? procedure : emptyProcedureExternalReadEntry;
        ListenerEntry listenerEntry = cache.registerDependencies(graph, entry, parent, listener, procedure_, false);
        if(entry.isReady()) entry.performFromCache(graph, procedure_);
        else {
          assert(entry.isPending());
          entry.compute(graph, procedure_);
          if(listenerEntry != null) cache.primeListenerEntry(listenerEntry, entry.getResult());
        }
    }
    
}
