/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.query;

import java.util.ArrayList;

import org.simantics.db.AsyncReadGraph;
import org.simantics.db.exception.DatabaseException;
import org.simantics.db.impl.BlockingAsyncMultiProcedure;
import org.simantics.db.impl.graph.ReadGraphImpl;
import org.simantics.db.procedure.AsyncMultiProcedure;
import org.simantics.db.request.AsyncMultiRead;
import org.simantics.db.request.RequestFlags;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final public class AsyncMultiReadEntry<T> extends CacheEntryBase<AsyncMultiProcedure<T>> {

    private static final Logger LOGGER = LoggerFactory.getLogger(AsyncMultiReadEntry.class);

    protected AsyncMultiRead<T> request;
    
    AsyncMultiReadEntry(AsyncMultiRead<T> request) {
    	this.request = request;
    }
    
    @Override
    int makeHash() {
    	return request.hashCode();
    }
    
    @Override
    public Object getOriginalRequest() {
        return request;
    }
    
    @Override
    public void discard() {
    	super.discard();
    	request = null;
    	setResult(null);
    }
    
    final synchronized public void finish(AsyncReadGraph graph) {
    	
    	assert(isPending());

        synchronized(this) {
        	setReady();
        }
        
    }

    final synchronized public void except(AsyncReadGraph graph, Throwable t) {

    	assert(isPending());

        synchronized(this) {
        	except(t);
        }
        
    }

    @SuppressWarnings("unchecked")
	final synchronized public void addOrSet(Object item) {

    	assert(isPending());
    	
        ArrayList<T> value = (ArrayList<T>)getResult(); 
        value.add((T)item);
        
    }
    
    @Override
    public void clearResult(QuerySupport support) {
    	setResult(new ArrayList<T>());
    }
    
    @Override
    final public Query getQuery() {
    	
        return new Query() {

			@Override
			public void recompute(ReadGraphImpl graph) {

				try {

					BlockingAsyncMultiProcedure<T> proc = new BlockingAsyncMultiProcedure<>(graph.asyncBarrier, graph, new AsyncMultiProcedure<T>() {

						@Override
						public void execute(AsyncReadGraph graph, T result) {
							addOrSet(result);
						}

						public void finished(AsyncReadGraph graph) {
							finish(graph);
						};

						@Override
						public void exception(AsyncReadGraph graph, Throwable t) {
							except(t);
						}

					}, request);

					request.perform(graph , proc);

					proc.get();

				} catch (Throwable t) {
					
					except(t);
					
				}
				
			}

			@Override
			public void removeEntry(QueryProcessor processor) {
		    	processor.cache.remove(AsyncMultiReadEntry.this);
			}

			@Override
			public int type() {
				return RequestFlags.INVALIDATE;
			}
			
			@Override
			public String toString() {
				if(request == null) return "DISCARDED";
				else return request.toString() + statusOrException;
			}
        	
        };
        
    }

	@SuppressWarnings("unchecked")
	@Override
	public Object performFromCache(ReadGraphImpl graph, AsyncMultiProcedure<T> proc) {

        if(isExcepted()) {

            try {
                proc.exception(graph, (Throwable)getResult());
            } catch (Throwable t) {
                LOGGER.error("performFromCache proc.exception failed", t);
            }

        } else {

            final ArrayList<T> values = (ArrayList<T>)getResult();
            for(T value : values) {
                try {
                    proc.execute(graph, value);
                } catch (Throwable t) {
                    LOGGER.error("performFromCache proc.execute failed", t);
                }
            }

            try {
                proc.finished(graph);
            } catch (Throwable t) {
                LOGGER.error("performFromCache proc.finished failed", t);
            }

        }
		
		return getResult();
		
	}
	
	@Override
	public String toString() {
		if(request == null) return "DISCARDED";
		else return request.toString() + statusOrException;
	}

	public Object compute(ReadGraphImpl graph, AsyncMultiProcedure<T> procedure) throws DatabaseException {
		return graph.processor.cache.performQuery(graph, request, this, procedure);
	}

}
