/*******************************************************************************
 * Copyright (c) 2007, 2018 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl.graph;

import java.util.Collection;
import java.util.concurrent.atomic.AtomicInteger;

import org.simantics.db.common.utils.Logger;
import org.simantics.db.exception.RuntimeDatabaseException;
import org.simantics.db.impl.query.CacheEntry;
import org.simantics.db.impl.query.QueryProcessor.AsyncBarrier;

final public class AsyncBarrierImpl extends AtomicInteger implements AsyncBarrier {

	private static final long serialVersionUID = 4724463372850048672L;

	static final int WAIT_TIME = 600;

	public static final boolean PRINT = false;

	final public AsyncBarrierImpl caller;

	public AsyncBarrierImpl(AsyncBarrierImpl caller, CacheEntry<?> entry) {
		super(0);
		this.caller = caller;
        if (BarrierTracing.BOOKKEEPING) {
            BarrierTracing.trace(this, entry);
        }
	}


	public void inc() {

	    if(BarrierTracing.BOOKKEEPING) {
	        BarrierTracing.inc(this);
	    } else {
	        inc(null, null);
	    }
	    
	}

	void inc(Object id, String info) {

		if(PRINT) {
			System.err.println("inc barrier[" + get() + "] " + this);
			StackTraceElement[] elems = new Exception().getStackTrace();
			for(int i=0;i<4;i++) System.err.println(elems[i]);
		}

		if (incrementAndGet() == 1) {
			if (caller != null) {
		        if(BarrierTracing.BOOKKEEPING) {
                    caller.inc(this, "Child");
		        } else {
                    caller.inc(null, null);
		        }
			}
		}

	}

	public void dec() {

		if(PRINT) {
			System.err.println("dec barrier[" + get() + "] " + this);
			StackTraceElement[] elems = new Exception().getStackTrace();
			for(int i=0;i<3;i++) System.err.println(elems[i]);
		}

		int count = decrementAndGet();
		if (count < 1) {
            if(BarrierTracing.BOOKKEEPING) {
                BarrierTracing.dec(this, count);
            }
			if (count == 0) {
				if (caller != null) {
					caller.dec();
				}
			}
			if (count < 0) {
				Logger.defaultLogError(
						"Database request processing error. The application code has performed illegal actions (probably called multiple times the execute or exception method of a single result request.",
						new Exception());
			}
			assert (count >= 0);
		}

	}

	public static String report(AsyncBarrierImpl barrier) {
		CacheEntry<?> e = BarrierTracing.entryMap.get(barrier);
		if(e != null) return e.toString();
		else return "Barrier@" + System.identityHashCode(barrier);
	}
	
	public static void printReverse(AsyncBarrierImpl barrier, int indent) {

		if (barrier.get() == 0)
			return;
		for (int i = 0; i < indent; i++)
			System.err.print(" ");
		System.err.println("[" + barrier.get() + " requests]: " + report(barrier));
//		if (BOOKKEEPING) {
//			Debugger debugger = debuggerMap.get(barrier);
//			debugger.toErr(indent + 2);
//		}

		Collection<AsyncBarrierImpl> children = BarrierTracing.reverseLookup.get(barrier);
		if (children != null) {
			for (AsyncBarrierImpl child : children)
				printReverse(child, indent + 2);
		}

	}

	public void waitBarrier(Object request, ReadGraphImpl impl) {

		if (get() > 0) {

			long waitCount = 0;

			while (get() != 0) {

				boolean executed = impl.performPending();
				if(executed) waitCount = 0;
				
				++waitCount;
				if(waitCount > 100) Thread.yield();
				if(waitCount > 1000) {
					try {
						Thread.sleep(1);
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
				}
				if(waitCount > WAIT_TIME*1000) {

					System.err.println("AsyncBarrierImpl.waitBarrier("
							+ request
							+ ") is taking long to execute, so far "
							+ (waitCount / 1000) + " s.");

					if (BarrierTracing.BOOKKEEPING) {
						synchronized (BarrierTracing.reverseLookup) {
							printReverse(this, 0);
						}
					}

//					if(Development.DEVELOPMENT) {
//						impl.processor.threadLocks[0].lock();
//						System.err.println("-queues=" + impl.processor.queues[0].size());
//						impl.processor.threadLocks[0].unlock();
//						System.err.println("-own=" + impl.processor.ownTasks[0].size());
//						System.err.println("-ownSync=" + impl.processor.ownSyncTasks[0].size());
//						for(SessionTask task : impl.processor.ownSyncTasks[0]) {
//							System.err.println("--" + task);
//						}
//					}

					throw new RuntimeDatabaseException("Request timed out.");
					//waitCount = 0;

				}

			}

		}

	}

	public void restart() {
		assertReady();
		if(BarrierTracing.BOOKKEEPING) {
		    BarrierTracing.restart(this);
		}
	}

	public void assertReady() {
		int current = get();
		if (current != 0)
			throw new AssertionError("Barrier was not finished (pending="
					+ current + ").");
	}

	public void report() {
		// System.out.println("Barrier log:");
		// for(Map.Entry<String, Integer> entry : sources.entrySet()) {
		// System.out.println(entry.getKey() + " " + entry.getValue());
		// }
		// System.out.println("SyncIntProcedure log:");
		// for(Map.Entry<String, Integer> entry :
		// SyncIntProcedure.counters.entrySet()) {
		// System.out.println(entry.getKey() + " " + entry.getValue());
		// }
	}

	@Override
	public String toString() {
		return report(this);
//		return "AsyncBarrierImpl@" + System.identityHashCode(this)
//				+ " - counter = " + get() + " - caller = " + caller;
	}

}
