/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.db.impl;

import org.simantics.db.exception.DatabaseException;

public class ClusterTraitsBase {
    private static final int RK_TYPE_BITS = 1; // bits 31-31, 0=database 1=virtual
    private static final int RK_CLUSTER_BITS = 19; // bits 30-12
    private static final int RK_CLUSTER_MAX = (1<<RK_CLUSTER_BITS)-1;
    private static final int RK_INDEX_BITS = 12; // bits 11-0, index zero reserved (not a legal resource index)
    private static final int RK_INDEX_MAX = (1<<RK_INDEX_BITS)-1;
    private static final int RID_TYPE_BITS = RK_TYPE_BITS; // bits 63-63, 0=database 1= virtual
    private static final int RID_CLUSTER_BITS = 64-RID_TYPE_BITS-RK_INDEX_BITS; // bits 62-12
    private static final int RID_CLUSTER_MAX = (1<<RID_CLUSTER_BITS)-1;
    private static final int RID_INDEX_BITS = RK_INDEX_BITS;
    private static final int RID_INDEX_MAX = (1<<RID_INDEX_BITS)-1;
//    private static final int RESOURCE_INDEX_BITS = 14; // resource index bits in cluster
//    private static final int RESOURCE_INDEX_MAX = (1<<RESOURCE_INDEX_BITS)-1;
    public static final boolean isIllegalResourceIndex(int resourceIndex) {
        return resourceIndex < 1 || resourceIndex > RK_INDEX_MAX;
    }
    public static final short getResourceIndexFromResourceKey(int resourceKey) throws DatabaseException {
        short resourceIndex = (short)(resourceKey & RK_INDEX_MAX);
        if (isIllegalResourceIndex(resourceIndex))
            throw new DatabaseException("Illegal resource key " + resourceKey);
        return resourceIndex;
    }
    public static final short getResourceIndexFromResourceKeyNoThrow(int resourceKey) {
        return (short)(resourceKey & RK_INDEX_MAX);
    }
    public static final int getClusterKeyFromResourceKey(int resourceKey)
    throws DatabaseException {
        int clusterKey = resourceKey >>> RK_INDEX_BITS;
        if (clusterKey < 1 || clusterKey >= getClusterArraySize())
            throw new DatabaseException("Illegal cluster key for resource key=" + resourceKey);
        return clusterKey;
    }
    public static final int getClusterKeyFromResourceKeyNoThrow(int resourceKey) {
        return resourceKey >>> RK_INDEX_BITS;
    }
    public static final boolean isVirtualClusterKey(int clusterKey) {
        return (clusterKey & ~RK_CLUSTER_MAX) != 0; 
    }
    // This requires that builtin cluster has cluster index 1 and that IIS entities
    // are given the first three indexes within that cluster.
    public static final int getCompleteTypeIntFromResourceKey(int resourceKey) {
        if ((resourceKey & 0xFFFFEFFC) != 0)
            return 0; // not complete
        return resourceKey & 3;
    }
    public static final ClusterI.CompleteTypeEnum getCompleteTypeFromResourceKey(int resourceKey) {
        return ClusterI.CompleteTypeEnum.make(getCompleteTypeIntFromResourceKey(resourceKey));
    }
    public static final int getCompleteTypeResourceKeyFromEnum(ClusterI.CompleteTypeEnum completeType)
    throws DatabaseException {
        switch (completeType) {
            default: throw new DatabaseException("Illegal compete type for getCompletePredicateKey: " + completeType);
            case InstanceOf: return 4097;
            case Inherits: return 4098; 
            case SubrelationOf: return 4099;
        }
    }
    public static final int createResourceKey(int clusterIndex, int resourceIndex)
    throws DatabaseException {
// Do not use assert because this method is called with the assumption that it
// will throw exception if arguments are not correct. If you really can't afford
// the few if tests then use the NoThrow version of this method.
        if (clusterIndex < 0 || clusterIndex > RK_CLUSTER_MAX)
            throw new DatabaseException("Illegal cluster index " + clusterIndex);
        if (isIllegalResourceIndex(resourceIndex))
            throw new DatabaseException("Illegal resource index " + resourceIndex);
        return createResourceKeyNoThrow(clusterIndex, resourceIndex);
    }
    public static final int createResourceKeyNoThrow(int clusterIndex, int resourceIndex){
        return (clusterIndex << RK_INDEX_BITS | resourceIndex);
    }
    public static final int getClusterBits(int clusterKey) {
        return clusterKey << RK_INDEX_BITS;
    }
    public static final boolean isCluster(int clusterKeyBits, int resourceKey) {
//        return ((resourceKey ^ clusterKeyBits) & RK_INDEX_MAX) == 0;
        return clusterKeyBits == (resourceKey & ~RK_INDEX_MAX);

    }
    public static final int getClusterMaskFromResourceKey(int resourceKey) {
        return resourceKey & ~RK_INDEX_MAX;
    }
    public static final int getMaxNumberOfResources() {
        return RK_INDEX_MAX;
    }
    public static final long createResourceId(long clusterId, int resourceIndex)
    throws DatabaseException {
        if (clusterId < 1 || (clusterId > RID_CLUSTER_MAX))
            throw new DatabaseException("Illegal cluster id " + clusterId);
        if (isIllegalResourceIndex(resourceIndex))
            throw new DatabaseException("Illegal resource index " + resourceIndex);
        return createResourceIdNoThrow(clusterId, resourceIndex);
    }
    public static final long createResourceIdNoThrow(long clusterId, int resourceIndex) {
        return (clusterId<<RID_INDEX_BITS | resourceIndex);
    }
    public static final long getClusterIdFromResourceId(long resourceId) {
        return resourceId >>> RID_INDEX_BITS;
    }
    public static final int getResourceIndexFromResourceId(long resourceId) {
        return (int)(resourceId & RID_INDEX_MAX);
    }
    // Max number of resident clusters per session.
    public static final int getClusterArraySize() {
        return Math.min(RK_CLUSTER_MAX, 1<<15);
    }
}