/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.type;

import java.util.Set;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.simantics.databoard.accessor.error.ReferenceException;
import org.simantics.databoard.accessor.reference.ChildReference;
import org.simantics.databoard.util.IdentityPair;
import org.simantics.databoard.util.Limit;
import org.simantics.databoard.util.Range;
import org.simantics.databoard.util.RangeException;

public class StringType extends Datatype {
		
	public static final String KEY_PATTERN = "pattern";
	public static final String KEY_MIMETYPE = "mimeType";
	public static final String KEY_LENGTH = "length";
	
    private transient Range _length;
    private transient String _lengthIsForStr;    
    private transient Pattern pattern_;
    private transient String pattern_IsForStr;
    
    public StringType () {}
    
    public StringType (String pattern) {
    	setPattern(pattern);
    }

    public StringType (String pattern, String mimeType, Range length) {
    	setPattern(pattern);
    	setMimeType(mimeType);
    	setLength(length);
    }

    public StringType (String pattern, String mimeType, String length) {
    	setPattern(pattern);
    	setMimeType(mimeType);
    	setLength(length);
    }
    
    @Override
    public int getComponentCount() {
    	return 0;
    }
    
    @Override
    public Datatype getComponentType(int index) {
    	throw new IllegalArgumentException();
    }
    
    @Override
    public Datatype getComponentType(ChildReference path) {
    	if (path==null) return this;
    	throw new IllegalArgumentException();
    }    
    
    @Override
    protected boolean deepEquals(Object obj, Set<IdentityPair<Datatype, Datatype>> compareHistory) {
		if (this==obj) return true;
		if ( !hasEqualMetadata(obj) ) return false;
		return obj instanceof StringType;
	}
    
	@Override
	public int hashCode() {		
		return 0x43676323 + super.hashCode(); 
	}    

	@Override
	public void accept(Visitor1 v, Object obj) {
	    v.visit(this, obj);        
	}

	@Override
	public <T> T accept(Visitor<T> v) {
	    return v.visit(this);
	}

	public String getPattern() {
		return metadata.get(KEY_PATTERN);
	}

	public void setPattern(String pattern) 
	throws PatternSyntaxException 
	{
		if ( pattern == null ) metadata.remove(KEY_PATTERN); else
		metadata.put(KEY_PATTERN, pattern);
	}

	public String getMimeType() {
		return metadata.get(KEY_MIMETYPE);
	}

	public void setMimeType(String mimeType) {
		if (mimeType==null) metadata.remove( KEY_MIMETYPE ); else
		metadata.put(KEY_MIMETYPE, mimeType);
	}

	public Range getLength() {
		String lengthStr = metadata.get( KEY_LENGTH );
		if ( lengthStr == null ) return null;
		if ( lengthStr == _lengthIsForStr ) return _length;
		try {
			_lengthIsForStr = lengthStr;
			_length = Range.valueOf( lengthStr );
		} catch (RangeException e) {
			_length = null;
		}
		return _length;
	}

	public int minLength() {
    	if (_length==null) return 0;
    	Limit l = _length.getLower();
    	int value = l.getValue().intValue();
    	if (l.isExclusive()) value++;
    	return value;
	}
	
	public int maxLength() {
    	if (_length==null) return Integer.MAX_VALUE;
    	Limit l = _length.getUpper();
    	int value = l.getValue().intValue();
    	if (l.isExclusive()) value--;
    	return value;
	}

	/**
	 * Get compiled reg exp pattern 
	 * 
	 * @return pattern or null
	 */
	public Pattern getCompiledPattern()
	{
		String patternStr = metadata.get( KEY_PATTERN );
		if ( patternStr == null ) return null;
		if ( patternStr == pattern_IsForStr ) return pattern_;
		try {
			pattern_IsForStr = patternStr;
			pattern_ = Pattern.compile(patternStr);
		} catch (PatternSyntaxException e) {
			pattern_ = null;
		}
		return pattern_;
	}

	public void setLength(String length) {
		if ( length == null ) metadata.remove( KEY_LENGTH ); else
		metadata.put( KEY_LENGTH, length );
	}
	
	public void setLength(Range range) {
		if (range == null) {
			this._length = null;
			this._lengthIsForStr = null;
			metadata.remove( KEY_LENGTH );
		} else {
			this._length = range;
			this._lengthIsForStr = range.toString();
			metadata.put( KEY_LENGTH, _lengthIsForStr );					
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public <T extends Datatype> T getChildType(ChildReference reference) throws ReferenceException {
		if (reference==null) return (T) this;
		throw new ReferenceException(reference.getClass()+" is not a subreference of StringType");	
	}
	
	
	
}
