package org.simantics.databoard.channel;

import java.io.IOException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.simantics.databoard.binding.Binding;

/**
 * Delayed response object. The result or error may be or become available to
 * this object at any given time.    
 * 
 * This interface is intended to be used by clients that make service requests
 * over command channel. 
 *
 * @author Toni Kalajainen <toni.kalajainen@iki.fi>
 */
public interface AsyncRequest {

	/**
	 * Get the current request status. <p>
	 *  
	 * If Succeeded or Failed, then result or error is available and can be read 
	 * without wait. <p>
	 * 
	 * Succeeded and Failed statuses are final. <p>
	 */
	enum Status {Waiting, Succeed, Failed} Status getStatus(); 			
	
	/**
	 * Get result if available. 
	 * 
	 * @param binding
	 * @return result or <tt>null</tt>
	 * @throws RequestException
	 */
	Object getResult(Binding binding) throws RequestException;
	
	/**
	 * Get error if available.
	 * 
	 * Cause is (typically) one of these:
	 *  o Communication problems {@link IOException}
	 *  o Timeout {@link TimeoutException}
	 *  o Service(handler) problem {@link ServiceException}
	 * 
	 * @return error or <tt>null</tt>
	 */
	Exception getError(); 	
	
	/**
	 * Synchronous wait for result until default timeout. 
	 * Default timeout is configured to the channel. 
	 * If timeout occurs TimeoutException is thrown wrapped in RequestException.
	 * 
	 * @param binding the format for the result
	 * @return the result object
	 * @throws RequestException
	 */
	Object waitForResult(Binding binding) throws RequestException;
	
	/**
	 * Wait for result or break after until timeout. 
	 * If timeout occurs TimeoutException is thrown wrapped in RequestException.
	 * 
	 * @param binding
	 * @param timeout
	 * @param unit
	 * @return result
	 * @throws RequestException
	 */
	Object waitForResult(Binding binding, long timeout, TimeUnit unit) throws RequestException;

	/**
	 * Set a listener. If the result is already available, the event
	 * schedueled immediately.
	 * 
	 * @param listener (listener may not block) or null to remove listener
	 */
	void setListener(RequestListener listener);
	
	interface RequestListener {
		
		/**
		 * Get thread where the listening is to be processed
		 *   
		 * @return thread to run listener events
		 */
		Executor getThread();
		
		/**
		 * Request completed, the result is available
		 * 
		 * @param result the result 
		 */
		void onCompleted(Object result);
		
		/**
		 * There was an error in processing the request
		 * 
		 * @param error the error
		 */
		void onError(ServiceException error);
		
	}
	
}
