/*******************************************************************************
 *  Copyright (c) 2010 Association for Decentralized Information Management in
 *  Industry THTH ry.
 *  All rights reserved. This program and the accompanying materials
 *  are made available under the terms of the Eclipse Public License v1.0
 *  which accompanies this distribution, and is available at
 *  http://www.eclipse.org/legal/epl-v10.html
 *
 *  Contributors:
 *      VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.databoard.binding.impl;

import org.simantics.databoard.binding.DoubleBinding;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.error.UnsupportedOperationException;
import org.simantics.databoard.type.DoubleType;

/**
 * Binds DoubleType to java.lang.Double
 *
 * @author Toni Kalajainen <toni.kalajainen@vtt.fi>
 */
public class DoubleBindingDefault extends DoubleBinding {

	public DoubleBindingDefault(DoubleType type) {
		super(type);
	}
	
	@Override
	public DoubleType type() {
		return (DoubleType) type;
	}
	
	public Object create(Double value) {
		return value;
	}

	public Object create(double value) {
		return Double.valueOf(value);
	}
	
	@Override
	public Object create(Number value) 
	{
		if (value.getClass()==Double.class) return value;
		return Double.valueOf( value.doubleValue() );
	}

	@Override
	public Object create(String value) throws BindingException {
		try {
			return Double.parseDouble(value);
		} catch (java.lang.NumberFormatException e) {
			throw new BindingException( e );
		}
	}
	
	public Double getValue(Object o) 
	throws BindingException
	{
		if (o.getClass()!=Double.class)
			throw new BindingException("Unexpected class "+o.getClass().getSimpleName()+", Double expected");
		return ((Double)o);
	}
	
	public double getValue_(Object o)
	throws BindingException
	{
		if (o.getClass()!=Double.class)
			throw new BindingException("Unexpected class "+o.getClass().getSimpleName()+", Double expected");
		return ((Double)o);
	}
	
	@Override
	public void setValue(Object obj, Number value) throws BindingException {
		throw new UnsupportedOperationException("Cannot change the value of immutable java.lang.Double");
	}
	
	public void setValue(Object obj, double value) throws BindingException {
		throw new UnsupportedOperationException("Cannot change the value of immutable java.lang.Double");
	}
	
    @Override
	public boolean isInstance(Object obj) {
		return obj instanceof Double;
	}
		
    @Override
    public boolean isImmutable() {
    	return true;
    }
    
    @Override
    public int compare(Object o1, Object o2)
    {
    	Double d1 = (Double) o1;
    	Double d2 = (Double) o2;
    	return d1.compareTo(d2);
    }
    
}

