package org.simantics.charts.export;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.simantics.Simantics;
import org.simantics.databoard.binding.error.BindingException;
import org.simantics.databoard.binding.mutable.Variant;
import org.simantics.databoard.util.Bean;
import org.simantics.db.exception.DatabaseException;
import org.simantics.export.core.ExportContext;
import org.simantics.export.core.error.ExportException;
import org.simantics.export.core.intf.ExportClass;
import org.simantics.export.core.manager.Content;
import org.simantics.history.History;
import org.simantics.history.HistoryException;
import org.simantics.history.HistoryManager;
import org.simantics.history.ItemManager;
import org.simantics.history.util.subscription.SamplingFormat;
import org.simantics.simulation.experiment.IExperiment;
import org.simantics.simulation.experiment.IHistoryExperiment;
import org.simantics.simulation.export.CSVItemsQuery;
import org.simantics.simulation.export.CSVItemsQuery.CSVItem;
import org.simantics.simulation.export.CSVWriter;
import org.simantics.simulation.export.ExperimentExportClass;
import org.simantics.simulation.project.ExperimentManager;
import org.simantics.simulation.project.IExperimentManager;
import org.simantics.utils.datastructures.MapList;

public class ExportChartCSV extends org.simantics.simulation.export.ExperimentExportClass implements ExportClass {

	
	@Override
	public void export(List<Content> contents, 
			Object writer_,
			ExportContext context, 
			Variant options,
			IProgressMonitor monitor, 
			MapList<Content, Content> attachmentMap
			) throws ExportException {
		
		// Flush all experiments, just in case.
		IExperimentManager em = Simantics.getProject().getHint( ExperimentManager.KEY_EXPERIMENT_MANAGER );
		if ( em != null ) {
			for (IExperiment exp : em.getExperiments()) {
				if ( exp instanceof IHistoryExperiment ) {
					IHistoryExperiment he = (IHistoryExperiment) exp;
					try {
						he.flushHistory();
					} catch (HistoryException e) {
					}
				}
			}
		}
		
		CSVWriter writer = (CSVWriter) writer_;
		
		List<HistoryManager> historiesToClose = new ArrayList<HistoryManager>(); 
		try {
			List<org.simantics.simulation.export.ExperimentExportClass.ModelRef> models = ExperimentExportClass.getResult(context, options, true);			

			// Label format is the following
			// "MODEL EXPERIMENT/RUN ITEM" 
			// MODEL is omited, if itemCount <= 1
			// RUN is omited, if runCount <= 1 
			int runCount = 0;
			int modelCount = models.size();
			for (ModelRef model : models) runCount += model.enabledRunCount();
			
			List<CSVItem> items = new ArrayList<CSVItem>();
			for (Content content : contents) {
				items.addAll( context.session.syncRequest( new CSVItemsQuery( content.url ) ) );
			}

			// 
			for (ModelRef model : models) {
				for (ExperimentRef experiment : model.experiments) {
					if ( experiment.runs.isEmpty() ) continue;					
					for (RunRef run : experiment.runs) {						
						if ( run.historyFolder == null || !run.historyFolder.exists() ) continue;
			
						HistoryManager history = History.openFileHistory( run.historyFolder );
						historiesToClose.add( history );
		        		ItemManager im = new ItemManager( history.getItems() );
						
						for ( CSVItem item : items ) {
							if ( !item.modelUri.equals( model.uri ) ) continue;
							StringBuilder labelBuilder = new StringBuilder();
							labelBuilder.append( item.label );
							if ( modelCount>1 ) {
								labelBuilder.append(" (");
								labelBuilder.append(model.label);
								labelBuilder.append(")");
							}
							if ( runCount>1 ) {
								labelBuilder.append(" (");
								labelBuilder.append( experiment.label );
								labelBuilder.append( "\\" );
								labelBuilder.append( run.label );
								labelBuilder.append(")");
							}
							
			            	List<Bean> historyItems = im.search("variableId", item.variableReference);
			            	Collections.sort(historyItems, SamplingFormat.INTERVAL_COMPARATOR);
			            	if (items.isEmpty()) continue;
			            	Bean config = historyItems.get(0);
			            	String historyId = (String) config.getFieldUnchecked("id");
														
							writer.addItem(history, historyId, labelBuilder.toString(), item.variableReference, item.unit);
						}
						
					}
				}
			}
					
		} catch (DatabaseException e) {
			throw new ExportException( e );
		} catch (HistoryException e) {
			throw new ExportException( e );
		} catch (BindingException e) {
			throw new ExportException( e );
		} finally {
			for (HistoryManager history : historiesToClose) history.close();
		}
		
	}

	@Override
	public List<String> validate(String contentUri, ExportContext context, Variant options) {
		return Collections.emptyList();
	}
	
}
