/*******************************************************************************
 * Copyright (c) 2010, 2011 Association for Decentralized Information Management in
 * Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.model.dnd;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;

import org.eclipse.swt.dnd.DND;
import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.model.InvalidContribution;
import org.simantics.browsing.ui.model.nodetypes.EntityNodeType;
import org.simantics.browsing.ui.model.nodetypes.NodeType;
import org.simantics.browsing.ui.model.nodetypes.OrderedNodeTypeMultiMap;
import org.simantics.db.ReadGraph;
import org.simantics.db.Resource;
import org.simantics.db.exception.DatabaseException;
import org.simantics.viewpoint.ontology.ViewpointResource;

public class DndBrowseContext {
    OrderedNodeTypeMultiMap<DropActionContribution> dropActionContributions = new OrderedNodeTypeMultiMap<DropActionContribution>();
    
    private DndBrowseContext() {
    }
    
    public static DndBrowseContext create(ReadGraph g, Collection<Resource> browseContextResources) throws DatabaseException, InvalidContribution {
        ViewpointResource vr = ViewpointResource.getInstance(g);
        DndBrowseContext browseContext = new DndBrowseContext();
        for(Resource browseContextResource : findSubcontexts(g, browseContextResources)) {
            for(Resource actionContributionResource : 
                g.getObjects(browseContextResource, vr.BrowseContext_HasDropActionContribution)) {
                DropActionContribution.load(g, actionContributionResource,
                        browseContext.dropActionContributions
                        );
            }
        }
        return browseContext;
    }
    
    private static Collection<Resource> findSubcontexts(ReadGraph g,
            Collection<Resource> browseContexts) throws DatabaseException {
        ViewpointResource vr = ViewpointResource.getInstance(g);
        HashSet<Resource> result = new HashSet<Resource>(browseContexts);
        ArrayList<Resource> stack = new ArrayList<Resource>(browseContexts);
        while(!stack.isEmpty()) {
            Resource cur = stack.remove(stack.size()-1);
            for(Resource sc : g.getObjects(cur, vr.BrowseContext_Includes))
                if(result.add(sc))
                    stack.add(sc);
        }
        return result;
    }    
    
    private static NodeType getNodeType(ReadGraph graph, NodeContext parent) throws DatabaseException {
        NodeType nodeType = parent.getConstant(NodeType.TYPE);
        if(nodeType == null) {            
            // TODO remove this code when root of model browser is fixed
            Object input = parent.getConstant(BuiltinKeys.INPUT);
            if(input instanceof Resource)
                nodeType = EntityNodeType.getNodeTypeFor(graph, (Resource)input);
        }
        return nodeType;
    }
    
    /**
     * Returns drop copy action.
     */
    public Runnable getAction(ReadGraph graph, NodeContext target, Object source) throws DatabaseException {
    	return getAction(graph, target, source, DND.DROP_COPY);
    }
    
    /**
     * Return drop operation for given operation. @See org.eclipse.swt.dnd.DND
     */
    public Runnable getAction(ReadGraph graph, NodeContext target, Object source, int operation) throws DatabaseException {
        NodeType nodeType = getNodeType(graph, target);
        if(nodeType == null)
            return null;
        for(DropActionContribution contribution : dropActionContributions.get(graph, nodeType)) {
            Runnable action = contribution.getAction(graph, target, source, operation);
            if(action != null)
                return action;
        }
        return null;
    }
}
