package org.simantics.browsing.ui.model.browsecontexts;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.common.NodeContextBuilder;
import org.simantics.browsing.ui.model.actions.ActionBrowseContext;
import org.simantics.browsing.ui.model.nodetypes.EntityNodeType;
import org.simantics.browsing.ui.model.nodetypes.NodeType;
import org.simantics.db.ReadGraph;
import org.simantics.db.RequestProcessor;
import org.simantics.db.Resource;
import org.simantics.db.common.procedure.adapter.TransientCacheListener;
import org.simantics.db.common.request.UnaryRead;
import org.simantics.db.exception.DatabaseException;

/**
 * Local utilities for BrowseContext/ActionBrowseContext internal use.
 * 
 * @author Tuukka Lehtonen
 */
public final class BrowseContexts {

	public static String[] toSortedURIs(ReadGraph graph, Collection<Resource> rs) throws DatabaseException {
		String[] result = new String[rs.size()];
		int i = 0;
		for (Resource r : rs) {
			String uri = graph.getPossibleURI(r);
			if (uri != null)
				result[i++] = uri;
		}
		if (i > 1)
			Arrays.sort(result, 0, i);
		return i < result.length ? Arrays.copyOf(result, i) : result;
	}

	/**
	 * Wrapper for SCL usage because SCL does not know effect <code>RequestProcessor</code>
	 * 
	 * @param graph
	 * @param uris
	 * @return
	 * @throws DatabaseException
	 */
	public static BrowseContext toBrowseContextG(ReadGraph graph, String[] uris) throws DatabaseException {
	    return toBrowseContext(graph, uris);
	}
	
	public static BrowseContext toBrowseContext(RequestProcessor processor, String[] uris) throws DatabaseException {
		return processor.syncRequest(new URIsToBrowseContext(Arrays.asList(uris)), TransientCacheListener.<BrowseContext>instance());
	}

	   /**
     * Wrapper for SCL usage because SCL does not know effect <code>RequestProcessor</code>
     * 
     * @param graph
     * @param uris
     * @return
     * @throws DatabaseException
     */
	public static ActionBrowseContext toActionBrowseContextG(ReadGraph graph, String[] uris) throws DatabaseException {
	    return toActionBrowseContext(graph, uris);
	}
	
	public static ActionBrowseContext toActionBrowseContext(RequestProcessor processor, String[] uris) throws DatabaseException {
		return processor.syncRequest(new URIsToActionBrowseContext(Arrays.asList(uris)), TransientCacheListener.<ActionBrowseContext>instance());
	}
	
	public static NodeContext getNodeContextForResource(Resource resource) {
		return NodeContextBuilder.buildWithInput(resource);
	}

	public static NodeContext getNodeContextWithTypeForResource(ReadGraph graph, Resource resource) {
		try {
			return NodeContextBuilder.buildWithData(
					NodeType.KEY_SEQUENCE,
					new Object[] { resource, EntityNodeType.getNodeTypeFor(graph, resource) });
		} catch (DatabaseException e) {
			return NodeContextBuilder.buildWithInput(resource);
		}
	}

	public static class URIsToBrowseContext extends UnaryRead<List<String>, BrowseContext> {

		public URIsToBrowseContext(List<String> parameter) {
			super(parameter);
		}

		@Override
		public BrowseContext perform(ReadGraph graph) throws DatabaseException {
			Collection<Resource> resources = graph.syncRequest( new URIsToResources(parameter) );
			return resources.isEmpty() ? null : graph.syncRequest( new BrowseContextRequest(resources) );
		}

	}

	public static class URIsToActionBrowseContext extends UnaryRead<List<String>, ActionBrowseContext> {

		public URIsToActionBrowseContext(List<String> parameter) {
			super(parameter);
		}

		@Override
		public ActionBrowseContext perform(ReadGraph graph) throws DatabaseException {
			Collection<Resource> resources = graph.syncRequest( new URIsToResources(parameter) );
			return resources.isEmpty() ? null : graph.syncRequest( new ActionBrowseContextRequest(resources) );
		}

	}

	public static class URIsToResources extends UnaryRead<List<String>, Collection<Resource>> {

		public URIsToResources(List<String> parameter) {
			super(parameter);
		}

		@Override
		public Collection<Resource> perform(ReadGraph graph) throws DatabaseException {
			Collection<Resource> result = new ArrayList<Resource>(parameter.size());
			for (String uri : parameter) {
				Resource r = graph.getPossibleResource(uri);
				if (r != null)
					result.add(r);
			}
			return result;
		}

	}

	public static class ResourcesToURIs extends UnaryRead<Collection<Resource>, List<String>> {

		public ResourcesToURIs(Collection<Resource> parameter) {
			super(parameter);
		}

		@Override
		public List<String> perform(ReadGraph graph) throws DatabaseException {
			List<String> result = new ArrayList<String>(parameter.size());
			for (Resource r : parameter) {
				String uri = graph.getPossibleURI(r);
				if (uri != null)
					result.add(uri);
			}
			return result;
		}

	}

}
