/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.browsing.ui.common.processors;

import java.util.Arrays;

import org.simantics.browsing.ui.BuiltinKeys;
import org.simantics.browsing.ui.GraphExplorer;
import org.simantics.browsing.ui.NodeContext;
import org.simantics.browsing.ui.NodeContext.QueryKey;
import org.simantics.browsing.ui.NodeQueryManager;
import org.simantics.browsing.ui.content.ComparableContext;
import org.simantics.browsing.ui.content.PrunedChildrenResult;

public class DefaultFinalChildrenProcessor extends AbstractNodeQueryProcessor<NodeContext[]> {

    private final GraphExplorer explorer;

    public DefaultFinalChildrenProcessor(GraphExplorer explorer) {
        this.explorer = explorer;
    }

    @Override
    public QueryKey<NodeContext[]> getIdentifier() {
        return BuiltinKeys.FINAL_CHILDREN;
    }

    @Override
    public NodeContext[] query(NodeQueryManager manager, NodeContext context) {
        int maxChildren = Integer.MAX_VALUE;
        if (explorer != null)
            maxChildren = explorer.getMaxChildren(manager, context);

        ComparableContext[] comparables = manager.query(context, BuiltinKeys.COMPARABLE_CHILDREN);
        if (comparables == null) {
            // Could not make the children comparable which means we cannot sort them.
            // Just take the pruned children and truncate them to the child node limit.

            PrunedChildrenResult pruned = manager.query(context, BuiltinKeys.PRUNED_CHILDREN);
            int length = Math.min(pruned.getPrunedChildren().length, maxChildren);
            if (length < pruned.getPrunedChildren().length) {
                NodeContext[] truncated = new NodeContext[length];
                System.arraycopy(pruned.getPrunedChildren(), 0, truncated, 0, length);
                return truncated;
            } else {
                return pruned.getPrunedChildren();
            }
        }

        // Optimize away unnecessary work and allocations for trivial 0/1 child cases.
        if (comparables.length == 0)
            return NodeContext.NONE;
        if (comparables.length == 1)
            return new NodeContext[] { comparables[0].getContext() };

        // Sort the comparable children and truncate them to the child node limit.
//        long startTime = System.nanoTime();
//        new Exception("FINAL CHILDREN for " + comparables.length + " comparables").printStackTrace();
        Arrays.sort(comparables);
        int length = Math.min(comparables.length, maxChildren);
        NodeContext[] result = new NodeContext[length];
        for (int i = 0; i < length; i++)
            result[i] = comparables[i].getContext();
//        long endTime = System.nanoTime();
//        System.out.println("ARRAY SORT: (" + maxChildren + "/" + comparables.length + "): " + (endTime-startTime)*1e-6 + " ms");
        return result;
    }

    @Override
    public String toString() {
        return "FinalChildrenProcessor";
    }


}