/**
 * Copyright (c) 2013, impossibl.com
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * Neither the name of impossibl.com nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package com.impossibl.postgres.jdbc;

import com.impossibl.postgres.protocol.PrepareCommand;
import com.impossibl.postgres.protocol.QueryCommand;
import com.impossibl.postgres.protocol.ServerObjectType;
import com.impossibl.postgres.types.Type;

import static com.impossibl.postgres.jdbc.ErrorUtils.chainWarnings;
import static com.impossibl.postgres.jdbc.Exceptions.INVALID_COMMAND_FOR_GENERATED_KEYS;
import static com.impossibl.postgres.jdbc.Exceptions.NOT_SUPPORTED;
import static com.impossibl.postgres.jdbc.Exceptions.NO_RESULT_COUNT_AVAILABLE;
import static com.impossibl.postgres.jdbc.Exceptions.NO_RESULT_SET_AVAILABLE;
import static com.impossibl.postgres.jdbc.SQLTextUtils.appendReturningClause;
import static com.impossibl.postgres.jdbc.SQLTextUtils.prependCursorDeclaration;
import static com.impossibl.postgres.protocol.QueryCommand.ResultBatch.releaseResultBatches;

import java.sql.BatchUpdateException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.Arrays;
import java.util.Collections;

import static java.util.Arrays.asList;

class PGSimpleStatement extends PGStatement {

  SQLText batchCommands;

  public PGSimpleStatement(PGConnectionImpl connection, int type, int concurrency, int holdability) {
    super(connection, type, concurrency, holdability, null, null);
  }

  SQLWarning prepare(SQLText sqlText) throws SQLException {

    if (resultSetConcurrency != ResultSet.CONCUR_READ_ONLY) {

      name = connection.getNextStatementName();

      cursorName = "cursor" + name;

      if (!prependCursorDeclaration(sqlText, cursorName, resultSetType, resultSetHoldability, connection.autoCommit)) {

        cursorName = name = null;
      }

    }

    PrepareCommand prep = connection.getProtocol().createPrepare(name, sqlText.toString(), Collections.<Type> emptyList());

    SQLWarning warningChain = connection.execute(prep, true);

    resultFields = prep.getDescribedResultFields();

    return warningChain;
  }

  boolean execute(SQLText sqlText) throws SQLException {

    if (name != null) {
      dispose(connection, ServerObjectType.Statement, name);
    }

    if (processEscapes) {
      SQLTextEscapes.processEscapes(sqlText, connection);
    }

    if (sqlText.getStatementCount() > 1) {

      return executeSimple(sqlText.toString());

    }
    else {

      SQLWarning prepWarningChain = prepare(sqlText);

      boolean res = executeStatement(name, Collections.<Type> emptyList(), Collections.<Object> emptyList());

      if (cursorName != null) {
        res = super.executeSimple("FETCH ABSOLUTE 0 FROM " + cursorName);
      }

      warningChain = chainWarnings(prepWarningChain, warningChain);

      return res;

    }
  }

  @Override
  public boolean execute(String sql) throws SQLException {
    checkClosed();

    SQLText sqlText = connection.parseSQL(sql);

    return execute(sqlText);
  }

  @Override
  public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
    checkClosed();

    SQLText sqlText = connection.parseSQL(sql);

    if (autoGeneratedKeys != RETURN_GENERATED_KEYS) {
      return execute(sqlText);
    }

    if (!appendReturningClause(sqlText)) {
      throw INVALID_COMMAND_FOR_GENERATED_KEYS;
    }

    execute(sqlText);

    generatedKeysResultSet = getResultSet();

    return false;
  }

  @Override
  public boolean execute(String sql, int[] columnIndexes) throws SQLException {
    checkClosed();

    throw NOT_SUPPORTED;
  }

  @Override
  public boolean execute(String sql, String[] columnNames) throws SQLException {
    checkClosed();

    SQLText sqlText = connection.parseSQL(sql);

    if (!appendReturningClause(sqlText, asList(columnNames))) {
      throw INVALID_COMMAND_FOR_GENERATED_KEYS;
    }

    execute(sqlText);

    generatedKeysResultSet = getResultSet();

    return false;
  }

  @Override
  public ResultSet executeQuery(String sql) throws SQLException {

    if (!execute(sql)) {
      throw NO_RESULT_SET_AVAILABLE;
    }

    return getResultSet();
  }

  @Override
  public int executeUpdate(String sql) throws SQLException {

    if (execute(sql)) {
      throw NO_RESULT_COUNT_AVAILABLE;
    }

    return getUpdateCount();
  }

  @Override
  public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {

    if (execute(sql, autoGeneratedKeys)) {
      throw NO_RESULT_COUNT_AVAILABLE;
    }

    return getUpdateCount();
  }

  @Override
  public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {

    if (execute(sql, columnIndexes)) {
      throw NO_RESULT_COUNT_AVAILABLE;
    }

    return getUpdateCount();
  }

  @Override
  public int executeUpdate(String sql, String[] columnNames) throws SQLException {

    if (execute(sql, columnNames)) {
      throw NO_RESULT_COUNT_AVAILABLE;
    }

    return getUpdateCount();
  }

  @Override
  public void addBatch(String sql) throws SQLException {
    checkClosed();

    SQLText sqlText = connection.parseSQL(sql);

    if (batchCommands == null) {
      batchCommands = sqlText;
    }
    else {
      batchCommands.addStatements(sqlText);
    }
  }

  @Override
  public void clearBatch() throws SQLException {

    batchCommands = null;
  }

  @Override
  public int[] executeBatch() throws SQLException {
    checkClosed();

    int[] counts = new int[0];
    int c = 0;
    try {

      warningChain = null;

      if (batchCommands == null) {
        return new int[0];
      }

      execute(batchCommands);
      counts = new int[resultBatches.size()];

      for (c = 0; c < resultBatches.size(); ++c) {

        QueryCommand.ResultBatch resultBatch = resultBatches.get(c);

        if (resultBatch.getCommand().equals("SELECT")) {
          throw new BatchUpdateException("SELECT in executeBatch", Arrays.copyOf(counts, c));
        }

        if (resultBatch.getRowsAffected() != null) {
          counts[c] = (int)(long)resultBatches.get(c).getRowsAffected();
        }
        else {
          counts[c] = Statement.SUCCESS_NO_INFO;
        }
      }

      return counts;
    }
    catch (BatchUpdateException bue) {
      throw bue;
    }
    catch (SQLException se) {
      throw new BatchUpdateException(Arrays.copyOf(counts, c), se);
    }
    finally {

      resultBatches = releaseResultBatches(resultBatches);
      batchCommands = null;
      command = null;
    }

  }

}
