/*******************************************************************************
 * Copyright (c) 2007, 2010 Association for Decentralized Information Management
 * in Industry THTH ry.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     VTT Technical Research Centre of Finland - initial API and implementation
 *******************************************************************************/
package org.simantics.utils.ui.dialogs;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

/**
 * This class is shows message in dialog. This class can be instantiated from
 * any thread.
 * 
 * @author Toni Kalajainen
 */
public class ShowMessage implements Runnable {

    public enum MessageType {
        ERROR, CONFIRM, INFORMATION, QUESTION, WARNING
    };

    private String title;

    private String message;

    private MessageType type;
    
    private Display display;

    public ShowMessage(String title, String message, MessageType type) {
        this.title = title;
        this.message = message;
        this.type = type;
        this.display = getDisplay();
        display.asyncExec(this);
    }

    public ShowMessage(Display display, String title, String message, MessageType type) {
        this.title = title;
        this.message = message;
        this.type = type;
        this.display = display;
        display.asyncExec(this);
    }

    public ShowMessage(String title, String message, MessageType type, boolean sync) {
        this.title = title;
        this.message = message;
        this.type = type;
        this.display = getDisplay();
        if (sync)
            display.syncExec(this);
        else
            display.asyncExec(this);
    }

    public ShowMessage(Display display, String title, String message, MessageType type, boolean sync) {
        this.title = title;
        this.message = message;
        this.type = type;
        this.display = display;
        if (sync)
            display.syncExec(this);
        else
            display.asyncExec(this);
    }
    
    public Display getDisplay() {
        if (display!=null) return display;
        Display d = Display.getCurrent();
        if (d!=null) return d;
        return Display.getDefault();        
    }
    
    public void run() {
        Shell shell = display.getActiveShell();
        if (type == MessageType.ERROR)
            MessageDialog.openError(shell, title, message);
        if (type == MessageType.CONFIRM)
            MessageDialog.openConfirm(shell, title, message);
        if (type == MessageType.INFORMATION)
            MessageDialog.openInformation(shell, title, message);
        if (type == MessageType.QUESTION)
            MessageDialog.openQuestion(shell, title, message);
        if (type == MessageType.WARNING)
            MessageDialog.openWarning(shell, title, message);        
    }

    public static void showError(Display display, String title, String message) {
        new ShowMessage(display, title, message, MessageType.ERROR);        
    }
    
    public static void showError(String title, String message) {
        new ShowMessage(title, message, MessageType.ERROR);        
    }
    
    public static void showStatus(IStatus status) {
    	MessageType type = MessageType.INFORMATION;
    	String title = "";
    	if (status.getSeverity() == IStatus.OK) {
    		type = MessageType.INFORMATION;
    		title = "Status OK";
    	}
    	if (status.getSeverity() == IStatus.WARNING) {
    		type = MessageType.WARNING;
    		title = "Status Warning";
    	}
    	if (status.getSeverity() == IStatus.ERROR) {
    		type = MessageType.ERROR;
    		title = "Status Error";
    	}
    	if (status.getSeverity() == IStatus.CANCEL) {
    		type = MessageType.INFORMATION;
    		title = "Status Cancel";
    	}
    	if (status.getSeverity() == IStatus.INFO) {
    		type = MessageType.INFORMATION;
    		title = "Status Info";
    	}    		
    	new ShowMessage(title, status.getMessage(), type);
    }
    
    public static void showWarning(Display display, String title, String message) {
        new ShowMessage(display, title, message, MessageType.WARNING);        
    }
    
    public static void showWarning(String title, String message) {
        new ShowMessage(title, message, MessageType.WARNING);        
    }

    public static void showInformation(Display display, String title, String message) {
        new ShowMessage(display, title, message, MessageType.INFORMATION);        
    }
    
    public static void showInformation(String title, String message) {
        new ShowMessage(title, message, MessageType.INFORMATION);        
    }
    

    
    public static void syncShowError(Display display, String title, String message) {
        new ShowMessage(display, title, message, MessageType.ERROR, true);        
    }
    
    public static void syncShowError(String title, String message) {
        new ShowMessage(title, message, MessageType.ERROR, true);        
    }
    
    public static void syncShowWarning(Display display, String title, String message) {
        new ShowMessage(display, title, message, MessageType.WARNING, true);        
    }
    
    public static void syncShowWarning(String title, String message) {
        new ShowMessage(title, message, MessageType.WARNING, true);        
    }

    public static void syncShowInformation(Display display, String title, String message) {
        new ShowMessage(display, title, message, MessageType.INFORMATION, true);        
    }
    
    public static void syncShowInformation(String title, String message) {
        new ShowMessage(title, message, MessageType.INFORMATION, true);        
    }
    
}
